#include "fileread.h"

#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<string.h>
#include<fcntl.h>
#include<errno.h>
#include<sys/mman.h>
#include<sys/param.h>
#include<sys/stat.h>


/*******************************************************************************
 * allocateBuffer takes a size and allocates a buffer of that size. The buffer
 * is allocated using mmap() and is madvise()d to use THP. Additionally, the
 * buffer is initialized with garbage data in a way that is is physically
 * allocated at the moment that function returns (no implicit allocation will
 * be performed at write access)
 *
 * @param size Size of the buffer (in bytes)
 * @return Pointer the the (physically) allocated buffer
 ******************************************************************************/
char *allocateBuffer(size_t size) {
  char *buf = mmap(NULL, size, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
  madvise(buf, size, MADV_HUGEPAGE);

  for(u_int64_t i = 0; i < size; i += sysconf(_SC_PAGESIZE)) {
    *(volatile char *)(buf + i) = 0x2a;
  }

  return buf;
}

/*******************************************************************************
 * readFileToBuffer reads the file at the specified path and writes its content
 * to the submitted buffer. If the size of the buffer is not sufficient, it
 * stops writing and writes an error message to stdout.
 *
 * @param filePath path of the file that should be read
 * @param buf Pointer to the buffer where the content of the file should be
 *        written to (in addition to the content, a line with the file name is
 *        inserted before)
 * @param bufSize Size of the buffer
 * @return Number of bytes written to the buffer
 ******************************************************************************/
size_t readFileToBuffer(char *filePath, char *buf, size_t bufSize) {
  size_t nBytesWritten = 0;

  size_t pathSize = strlen(filePath) + 2;
  if(bufSize < pathSize) {
    dprintf(2, "Unable to write the content of %s to buffer, no space left.\n", filePath);
    errno = 0;
    return nBytesWritten;
  }

  sprintf(buf, "\r%s\n", filePath);
  nBytesWritten += pathSize;
  buf += nBytesWritten;

  int fd = open(filePath, O_RDONLY);
  if(fd == -1) {
    dprintf(2, "Unable to open %s. Error: %s\n", filePath, strerror(errno));
    errno = 0;
    return nBytesWritten;
  }

  ssize_t readBytes = 0;
  size_t readChunk = 1024;

  if(bufSize - nBytesWritten < readChunk) {
    dprintf(2, "Unable to write the complete content of %s to buffer, no space left.\n", filePath);
    errno = 0;
    return nBytesWritten;
  }

  while((readBytes = read(fd, buf, readChunk)) > 0) {
    nBytesWritten += readBytes;
    buf += readBytes;

    if(bufSize - nBytesWritten < readChunk) {
      dprintf(2, "Unable to write the complete content of %s to buffer, no space left.\n", filePath);
      errno = 0;
      return nBytesWritten;
    }
  }

  if(readBytes != 0) {
    dprintf(2, "Unable to read %s. Error: %s\n", filePath, strerror(errno));
    errno = 0;
  }

  if(close(fd) != 0) {
    dprintf(2, "Unable to close %s. Error: %s\n", filePath, strerror(errno));
    errno = 0;
  }

  return nBytesWritten;
}

/*******************************************************************************
 * readFilesToBuffer takes a buffer where the content of the files should be
 * stored and a list of files which should be stored in the buffer. The data
 * is stored in a way that a line with "filepath" is added before the actual
 * content of the corresponding file is written to the buffer so the content
 * can be matched to a file path afterwards.
 *
 * @param buf Pre-allocated buffer where the content can be stored
 * @param bufSize Size of the buffer (in bytes). Everything that does not fit in
 *        the buffer will be ignored to avoid re-allocation. A message is shown
 *        in stderr in that case.
 * @param fileList List of file paths that should be read and written to buf
 * @param nFiles Number of entries in the file path list
 * @return 0 is returned on success, -1 on error.
 ******************************************************************************/
int readFilesToBuffer(char *buf, size_t bufSize, char **fileList, int nFiles) {
  size_t currentOffset = 0;
  for(u_int64_t i = 0; i < nFiles; i++) {
    currentOffset += readFileToBuffer(fileList[i], buf + currentOffset, bufSize - currentOffset);
  }

  return 0;
}

/*******************************************************************************
 * reduceToRelativePath takes a relative path and a base path that should be
 * removed from the start of the relative path. Note that this function does not
 * check if the base path is really a part of the submitted path but instead
 * just removes strlen(base) from the head of path.
 *
 * @param path Full path that should be converted to a relative path from base
 * @param base Path that should be removed from the head of path
 * @return Path that is reduced to relative path
 ******************************************************************************/
char *reduceToRelativePath(char *path, char *base) {
  if(strlen(path) <= strlen(base) + 1) {
    return path;
  }

  return path + strlen(base) + 1;
}

/*******************************************************************************
 * createDirectoryRecursive takes a path and recursively creates the directory
 * by checking if the current part of the directory is already existent. If that
 * is not the case, the last part (e.g. after the last slash) is removed and
 * the function is called recursive with the new directory (one layer above).
 *
 * @param path Path that should be created
 * @return 0 is returned on success, -1 on error.
 ******************************************************************************/
int createDirectoryRecursive(char *path) {
  if (strlen(path) == 0) {
    return 0;
  }

  struct stat dirStat;

  if (stat(path, &dirStat) == 0) {
    return 0;
  }

  // Truncate the path at the last slash to get the part without the last dir
  ssize_t lastSlashIdx = -1;
  for (u_int64_t i = 0; i < MAXPATHLEN && path[i] != 0; i++) {
    if(path[i] == '/') {
      lastSlashIdx = i;
    }
  }

  if(lastSlashIdx > 0) {
    path[lastSlashIdx] = 0;

    // Create the part without last dir
    if (createDirectoryRecursive(path) != 0) {
      return -1;
    }
  }

  // Restore last part of the path
  if(lastSlashIdx > 0) {
    path[lastSlashIdx] = '/';
  }

  if(mkdir(path, 0755) == -1) {
    dprintf(2, "Unable to create directory %s. Error: %s\n", path, strerror(errno));
    errno = 0;
    return -1;
  }

  return 0;
}

/*******************************************************************************
 * createDirectoryForFile takes a path to a file (including the file name),
 * strips the file name and calls createDirectoryRecursive. Afterwards, the
 * part of the file name is added to the path again (in-place) to not modify
 * the submitted string.
 *
 * @param path Path of the file that should be created
 * @return 0 is returned on success, -1 on error.
 ******************************************************************************/
int createDirectoryForFile(char *path) {
  // Truncate the path at the last slash to get only the directory part (without
  // file name)
  ssize_t lastSlashIdx = -1;
  for(u_int64_t i = 0; i < MAXPATHLEN && path[i] != 0; i++) {
    if(path[i] == '/') {
      lastSlashIdx = i;
    }
  }

  if(lastSlashIdx > 0) {
    path[lastSlashIdx] = 0;
  }

  int retVal = createDirectoryRecursive(path);

  if(lastSlashIdx > 0) {
    path[lastSlashIdx] = '/';
  }

  return retVal;
}

/*******************************************************************************
 * persistBuffer takes a buffer that was filled with content using
 * readFilesToBuffer() and creates the file structures under the specified
 * targetDir.
 *
 * @param buf Pointer to the buffer that contains the actual data
 * @param targetDir Path to the directory where the initial data structure
 *        should be recreated.
 * @param baseDir Heading part of the file paths that should be ignored for the
 *        path recreation. The intention is to use the tool like cp -r which
 *        recreates only the part within the base dir without creating the
 *        entire base dir within the target dir. For that reason, the lenght of
 *        the initially submitted base dir might be removed from the paths.
 * @return 0 is returned on success, -1 on error.
 ******************************************************************************/
int persistBuffer(char *buf, char *targetDir, char *baseDir) {
  char currentPath [MAXPATHLEN];
  char *line = strtok(buf, "\n");
  int fd = -1;

  while(line != NULL) {
    // New file string detected (which is encoded by a heading '\r')
    if(line[0] == '\r') {
      // Try to close currently open file, if any file is open
      if (fd != -1) {
        if(close(fd) != 0) {
          dprintf(2, "Unable to close file %s. Error: %s\n", currentPath, strerror(errno));
          errno = 0;
        }
      }
      
      // Calculate relative path
      char *tmpPath = reduceToRelativePath(line + 1, baseDir);
      snprintf(currentPath, MAXPATHLEN, "%s/%s", targetDir, tmpPath);

      // Try to create relative directory structure
      if (createDirectoryForFile(currentPath) != 0) {
        fd = -1;
        line = strtok(NULL, "\n");
        continue;
      }

      // Try to open or create the requested file
      fd = open(currentPath, O_WRONLY|O_CREAT|O_TRUNC, 0644);
      if(fd == -1) {
        dprintf(2, "Unable to open file %s. Error: %s\n", currentPath, strerror(errno));
        errno = 0;
      }

      // Proceed with the next line
      line = strtok(NULL, "\n");
      continue;
    }

    if(fd == -1) {
      dprintf(2, "Ignoring line because currently no file is open. This should not happen.\n");

      // Proceed with the next line
      line = strtok(NULL, "\n");
      continue;
    }

    // Write current line to file
    dprintf(fd, "%s\n", line);

    // Proceed with the next line
    line = strtok(NULL, "\n");
  }

  if(fd != -1) {
    // Try to close currently open file, if any file is open
    if (fd != -1) {
      if(close(fd) != 0) {
        dprintf(2, "Unable to close file %s. Error: %s\n", currentPath, strerror(errno));
        errno = 0;
        return -1;
      }
    }
  }

  return 0;
}
