#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<malloc.h>
#include<sys/mman.h>
#include<math.h>
#include<errno.h>
#include<string.h>
#include<sys/wait.h>
#include<time.h>

#include "measureTiming.h"
#include "asm.h"
#include "memlib/memoryInspect.h"

/**
 * gcd returns the greatest common divisor of two values
 *
 * @param a First value
 * @param b Second value
 * @return greatest common divisor of a and b
 */
int gcd(u_int64_t a, u_int64_t b) {
    if(a==0) {
        return b;
    }
    return gcd(b%a, a);
}

/*******************************************************************************
 * printMappingsAndTime prints a graph that shows the mappings (e.g. mark blocks
 * of continuous PFNs) and the access time the allocation (e.g. the time the
 * first write access to the mapping took).
 *
 * @param aInfo: List of addrInfo items that contains physical mapping
 *        information
 * @param timings: List of access times that were measured for the first write
 *        access
 * @param len: Number of items in the aInfo and timings lists (which are the
 *        same for both lists, so one variable is sufficient).
 ******************************************************************************/
void printMappingsAndTime(addrInfo **aInfo, u_int64_t *timings, int len) {
    int digits = (int)ceil(log10(len * 1.0));

    int max = 0;
    for(int i = 0; i < len; i++) {
        if(max < timings[i]) {
            max = timings[i];
        }
    }

    int timeDigits = (int)ceil(log10(max * 1.0));

    int factor = max / 200;
    if(factor == 0) {
        factor = 1;
    }

    u_int64_t lastAddr = 0;

    int colorSpecifier = 8;
    for(int i = 0; i < len; i++) {
        if(aInfo[i]->pfn != lastAddr + 1) {
            // New group
            colorSpecifier++;
            if(colorSpecifier == 15) {
                colorSpecifier = 8;
            }
        }

        printf("\033[1m\033[38;5;%dm[%0*d]\033[0m PFN: 0x%016lx %0*ld: %.*s\n", colorSpecifier, digits, i, aInfo[i]->pfn, timeDigits, timings[i], (int)(timings[i]/factor), "##########################################################################################################################################################################################################");
        lastAddr = aInfo[i]->pfn;
    }
    printf("\033[0m");
}

/*******************************************************************************
 * printMappings prints a graph that shows the mappings (e.g. mark blocks of
 * continuous PFNs).
 *
 * @param aInfo: List of addrInfo items that contains physical mapping
 *        information.
 * @param len: Number of items in the timings list.
 ******************************************************************************/
void printMappings(addrInfo **aInfo, int len) {
    int digits = (int)ceil(log10(len * 1.0));
    for(int i = 0; i < len; i++) {
        printf("\t[%0*d] HVA: 0x%lx PFN: 0x%lx\n", digits, i, aInfo[i]->hva, aInfo[i]->pfn);
    }
}

/*******************************************************************************
 * printTimings prints a graph that shows the access time of the allocation
 * (e.g. the time the first write access to the mapping took).
 *
 * @param timings: List of access times that were measured for the first write
 *        access
 * @param len: Number of items in the timings list.
 ******************************************************************************/
void printTimings(u_int64_t *timings, int len) {
    int digits = (int)ceil(log10(len * 1.0));

    int max = 0;
    for(int i = 0; i < len; i++) {
        if(max < timings[i]) {
            max = timings[i];
        }
    }

    int timeDigits = (int)ceil(log10(max * 1.0));

    int factor = max / 200;
    if(factor == 0) {
        factor = 1;
    }

    printf("Max=%d Factor=%d\n", max, factor);

    for(int i = 0; i < len; i++) {
        printf("[%0*d] %0*ld: %.*s\n", digits, i, timeDigits, timings[i], (int)(timings[i]/factor), "##########################################################################################################################################################################################################");
    }
}

/*******************************************************************************
 * dumpPage takes a pointer to a sysconf(_SC_PAGESIZE) sized buffer and prints
 * the content of the buffer in hexadecimal characters.
 *
 * @param page: Pointer to the page that should be dumped.
 ******************************************************************************/
void dumpPage(void *page) {
    for(int i = 0; i < sysconf(_SC_PAGESIZE); i++) {
        printf("%02x ", ((char*)page)[i]);
        if((i+1) % 8 == 0) {
            printf("  ");
        }
        if((i+1) % 64 == 0) {
            printf("\n");
        }
    }
    printf("\n");
}

/*******************************************************************************
 * updatePhysicalInformation takes a list of addrInfo items and adds physical
 * address information (e.g. PFN) using /proc/self/pagemap.
 *
 * @param aInfo: addrInfo structure where the physical address information
 *        should be added
 * @param len: Number of the elements in the aInfo list
 * @return Pointer to the list with modified items
 ******************************************************************************/
addrInfo **updatePhysicalInformation(addrInfo **aInfo, int len) {
    if(addHpaToHva("/proc/self/pagemap", aInfo, len) != 0) {
        printf("Unable to get physical addresses.\n");
    }
    return aInfo;
}

/*******************************************************************************
 * printHelp takes the name of the binary and an exit code, prints the usage
 * page of the program and exits afterwards using the exit code.
 *
 * @param binary: Name of the binary that was executed (e.g. argv[0]).
 * @param exitCode: Code that should be used as exit code when the execution is
 *        ended after the help message is printed.
 ******************************************************************************/
void printHelp(char *binary, int exitCode) {
    printf("Usage: %s [-n nPages] [-g] [-d] [-h]\n", binary);
    printf("\tnPages number of 4K Pages to allocate (default: 512)\n");
    printf("\t-g Use gettime() instead of the rdtscp instruction\n");
    printf("\t-p Access PFNs as well\n");
    printf("\t-d Enable debug mode\n");
    printf("\t-h: Print this help message \n");
    exit(exitCode);
}

int main(int argc, char *const argv[]) {
    int nPages = 512;
    int debug = 0;
    int accessPFN = 0;
    int gettime = 0;

    int opt;
    extern char *optarg;
    while((opt = getopt(argc, argv, "n:gphd")) != -1) {
        switch(opt) {
            case 'n':
                nPages = atoi(optarg);
                if(nPages == 0) {
                    printf("Invalid value %s for nPages.\n", optarg);
                    printHelp(argv[0], EXIT_FAILURE);
                }
                break;
            case 'g':
                gettime = 1;
                break;
            case 'p':
                accessPFN = 1;
                break;
            case 'd':
                debug = 1;
                break;
            case 'h':
                printHelp(argv[0], EXIT_SUCCESS);
                break;
            default:
                printHelp(argv[0], EXIT_FAILURE);
                break;
        }
    }

    int64_t hvas[nPages];
    u_int64_t times[nPages];

    printf("[INFO]: Allocating %d 4K Pages\n", nPages);
    int pageSize = sysconf(_SC_PAGESIZE);

    struct timespec t1, t2;
    u_int64_t begin, end;

    for(int i = 0; i < nPages; i++) {
        char *ptr = NULL;
        posix_memalign((void**)&ptr, pageSize, pageSize * sizeof(char) - 8);

        if(gettime) {
            clock_gettime(CLOCK_MONOTONIC, &t1);
        } else {
            begin = rdtscp();
        }

        *(volatile char*)(ptr) = 0x2a;

        if(gettime) {
            clock_gettime(CLOCK_MONOTONIC, &t2);
        } else {
            end = rdtscp();
        }

        hvas[i] = (int64_t)(ptr);
        if(gettime) {
            times[i] = (t2.tv_sec - t1.tv_sec) * 1000000000 + (t2.tv_nsec - t1.tv_nsec);
        } else {
            times[i] = end - begin;
        }
    }

    addrInfo **aInfo = getAddrInfoFromHva(hvas, nPages);

    if(debug) {
      if(accessPFN) {
        updatePhysicalInformation(aInfo, nPages);
        printMappingsAndTime(aInfo, times, nPages);
      } else {
        printTimings(times, nPages);
      }
    }

    u_int64_t gcdVal = times[0];
    for(u_int64_t i = 1; i < nPages; i++) {
      gcdVal = gcd(gcdVal, times[i]);
    }

    if(gettime) {
      printf("[INFO]: Resolution of gettime: %ld\n", gcdVal);
    } else {
      printf("[INFO]: Resolution of rdtscp: %ld\n", gcdVal);
    }

    return EXIT_SUCCESS;
}
