#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<fcntl.h>
#include<getopt.h>
#include<errno.h>
#include<time.h>
#include<sys/wait.h>
#include<sys/mman.h>

#include "memlib/memoryInspect.h"
#include "memlib/util.h"
#include "measureRemap.h"

/*******************************************************************************
 * getRandomIntWithinBounds takes a lower and an upper bound and returns a
 * random integer between these bounds, including the lower and excluding the
 * upper bound.
 *
 * @param min: lower bound
 * @param max: upper bound
 * @return Random integer between min and max
 ******************************************************************************/
u_int64_t getRandomIntWithinBounds(u_int64_t min, u_int64_t max) {
  return rand() % (max - min) + min;
}

/*******************************************************************************
 * shuffleAInfoArray takes an array of addrInfo structs and shuffles it.
 *
 * @param aInfo: Array that should be shuffled
 * @param len: Number of elements in aInfo
 * @return Pointer to the suffled array
 ******************************************************************************/
addrInfo **shuffleAInfoArray(addrInfo **aInfo, u_int64_t len) {
  u_int64_t j;
  for(u_int64_t i = 0; i < len - 1; i++) {
    j = getRandomIntWithinBounds(i, len);
    addrInfo *tmp = aInfo[i];
    aInfo[i] = aInfo[j];
    aInfo[j] = tmp;
  }

  return aInfo;
}

/*******************************************************************************
 * getSectionNames accesses /proc/self/maps and dumps the names of the mappings
 * in a unified way, e.g. when there are mutliple mappings with the same name,
 * the returned list contains that name only once.
 *
 * @param len: Pointer to store the number of sections found
 * @return Pointer to the list of sections (with len entries)
 ******************************************************************************/
char **getSectionNames(u_int64_t *len) {
  *len = 0;
  char **sectionNames = NULL;

  char *buf = readFile("/proc/self/maps");
  char *line = strtok(buf, "\n");

  while(line != NULL) {
    u_int64_t i = 0;
    // end of address block
    while(line[i] != ' ') {
      i++;
    }
    i++;

    // end of permissions block
    while(line[i] != ' ') {
      i++;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ') {
      i++;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ') {
      i++;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ') {
      i++;
    }
    i++;

    //end of spaces block
    while(line[i] == ' ') {
      i++;
    }

    u_int64_t found = 0;
    for(u_int64_t j = 0; j < *len; j++) {
      u_int64_t s = strlen(line + i);
      if(strlen(sectionNames[j]) < s) {
        s = strlen(sectionNames[j]);
      }
      s += 1;

      if(strncmp(line + i, sectionNames[j], s) == 0) {
        found = 1;
        break;
      }
    }

    if(found == 0) {
      char *name = malloc(sizeof(char *) * (strlen(line + i) + 1));
      name = strncpy(name, line + i, strlen(line + i));
      name[strlen(line + i)] = 0;

      *len += 1;
      sectionNames = realloc(sectionNames, sizeof(char *) * *len);
      sectionNames[*len -1] = name;
    }

    line = strtok(NULL, "\n");
  }

  free(buf);
  return sectionNames;
}

/*******************************************************************************
 * getMappedPages accesses /proc/self/maps and returns the pointers of the pages
 * (one per 4K Page) that are mapped under the coresponding name
 *
 * @param len: Pointer to store the number of items in the returned array
 * @param sectionName: Name of the section for which the virtual addresses
 *        should be returned
 * @return Array of pointers that are mapped under the submitted name
 ******************************************************************************/
u_int64_t *getMappedPages(u_int64_t *len, char *sectionName) {
  u_int64_t *mappedPages = NULL;
  *len = 0;

  char *buf = readFile("/proc/self/maps");
  char *line = strtok(buf, "\n");
  u_int64_t startAddr = 0, endAddr = 0;

  while(line != NULL) {
    char *oLine = malloc(sizeof(char) * (strlen(line) + 1));
    strcpy(oLine, line);
    char *start = line;
    u_int64_t i = 0;
    while(line[i] != '-' && line[i] != 0) {
      i++;
    }
    line[i] = 0;
    i++;

    char *end = line + i;
    while(line[i] != ' ' && line[i] != 0) {
      i++;
    }
    if(line[i] == 0) {
      line = strtok(NULL, "\n");
      continue;
    }
    line[i] = 0;

    i++;

    // end of permissions block
    while(line[i] != ' ' && line[i] != 0) {
      i++;
    }
    if(line[i] == 0) {
      line = strtok(NULL, "\n");
      continue;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ' && line[i] != 0) {
      i++;
    }
    if(line[i] == 0) {
      line = strtok(NULL, "\n");
      continue;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ' && line[i] != 0) {
      i++;
    }
    if(line[i] == 0) {
      line = strtok(NULL, "\n");
      continue;
    }
    i++;

    // end of unknown block
    while(line[i] != ' ' && line[i] != 0) {
      i++;
    }
    if(line[i] == 0) {
      line = strtok(NULL, "\n");
      continue;
    }
    i++;

    //end of spaces block
    while(line[i] == ' ') {
      i++;
    }

    int nChars = strlen(sectionName);
    if(strlen(line + i) != nChars) {
      line = strtok(NULL, "\n");
      continue;
    }

    if(nChars > 0 && strncmp(line + i, sectionName, nChars) != 0) {
      line = strtok(NULL, "\n");
      continue;
    }

    startAddr = hexStrToInt(start);
    endAddr = hexStrToInt(end);

    for(u_int64_t page = startAddr; page < endAddr; page += sysconf(_SC_PAGESIZE)) {
      *len = *len + 1;
      mappedPages = realloc(mappedPages, (*len) * sizeof(u_int64_t));
      mappedPages[(*len)-1] = page;
    }

    line = strtok(NULL, "\n");
  }

  free(buf);
  return mappedPages;
}

/*******************************************************************************
 * updatePhysicalInformation takes a list of addrInfo items ad adds physical
 * address information using /proc/self/pagemap.
 *
 * @param aInfo: List of addrInfo items
 * @param len: Number of items in the aInfo list
 * @return List of addrInfo items with physical information.
 ******************************************************************************/
addrInfo **updatePhysicalInformation(addrInfo **aInfo, u_int64_t len) {
  if(addHpaToHva("/proc/self/pagemap", aInfo, len) != 0) {
    dprintf(2, "Unable to get physical addresses.\n");
  }
  return aInfo;
}

/*******************************************************************************
 * exportPages takes a filename and exports physical addresses of a addrInfo
 * list to a file.
 *
 * @param filename: Name of the file the addresses should be exported to
 * @param name: Name of the mapping section (is written to the file above the
 *        addresses).
 * @param aInfo: List of addrInfo items that should be used to store the PFNs
 * @param len: Number of items in the aInfo list
 ******************************************************************************/
void exportPages(char *filename, char *name, addrInfo **aInfo, u_int64_t len) {
  int fd = open(filename, O_CREAT|O_APPEND|O_WRONLY, 0644);
  if(fd < 0) {
    dprintf(2, "Unable to open %s. Error: %s\n", filename, strerror(errno));
    return;
  }

  dprintf(fd, "%s\n", name);
  for(u_int64_t i = 0; i < len; i++) {
    if(aInfo[i]->pfn == 0) {
      continue;
    }
    dprintf(fd, "0x%lx\n", aInfo[i]->pfn);
  }
  close(fd);
}

/*******************************************************************************
 * printHelp prints the usage page of the program and exits using a submitted
 * exit code
 *
 * @param binary: Name of the executable, e.g. argv[0]
 * @param exitCode: Code that should be used to exit
 ******************************************************************************/
void printHelp(char *binary, int exitCode) {
  printf("Usage: %s [-a] [-n nPages] [-f nFreePages] [-m nMapPagesInChild] [-h]\n", binary);
  printf("\t-a: Run in after-exec mode (compare pages loaded from tmpfile)\n");
  printf("\t-n: Specify the amout of pages to be mapped (default 1024)\n");
  printf("\t-f: Specify the amout of pages to be freed out of the mapped area (default 64)\n");
  printf("\t-m: Specify the amout of pages to be mappend in the child (default 1024)\n");
  printf("\t-r: Specify the amout of pages to be allocated before the pages are freed and the child is started (default 0)\n");
  printf("\t-h: Show this help message\n");
  exit(exitCode);
}

int main(int argc, char *const argv[]) {
  srand(time(NULL));

  u_int64_t afterExec = 0;
  u_int64_t nPages = 1024;
  u_int64_t nFreePages = 64;
  u_int64_t nPagesChild = 1024;
  u_int64_t nPagesRealloc = 0;

  int opt;
  extern char *optarg;
  while((opt = getopt(argc, argv, "an:f:m:r:h")) != -1) {
    switch(opt) {
      case 'a':
        afterExec=1;
        break;
      case 'n':
        nPages = atoi(optarg);
        if(nPages == 0) {
          dprintf(2, "The number of pages (-n %s) is invalid.\n", optarg);
          printHelp(argv[0], EXIT_FAILURE);
        }
      case 'f':
        nFreePages = atoi(optarg);
        if(nPages == 0) {
          dprintf(2, "The number of to free pages (-f %s) is invalid.\n", optarg);
          printHelp(argv[0], EXIT_FAILURE);
        }
        break;
      case 'm':
        nPagesChild = atoi(optarg);
        if(nPagesChild == 0) {
          dprintf(2, "The number of pages (-m %s) is invalid.\n", optarg);
          printHelp(argv[0], EXIT_FAILURE);
        }
        break;
      case 'r':
        nPagesRealloc = atoi(optarg);
        if(nPagesRealloc == 0) {
          dprintf(2, "The number of pages (-r %s) is invalid.\n", optarg);
          printHelp(argv[0], EXIT_FAILURE);
        }
        break;
      case 'h':
        printHelp(argv[0], EXIT_SUCCESS);
        break;
      default:
        dprintf(2, "Unknown option '%c'.\n", opt);
        printHelp(argv[0], EXIT_FAILURE);
        break;
    }
  }

  u_int64_t nDigitsChild = 0;
  u_int64_t remainder = nPagesChild;
  while(remainder > 0) {
    remainder /= 10;
    nDigitsChild++;
  }

  char *baseChildParamStr = "-n ";
  char *childParamStr = malloc(sizeof(char) * (strlen(baseChildParamStr) + nDigitsChild + 1));
  sprintf(childParamStr, "%s%ld", baseChildParamStr, nPagesChild);


  int64_t *hvas = malloc(sizeof(int64_t) * nPages);

  if(afterExec) {
    dprintf(2, "[INFO]: Child: Allocating %ld Pages\n", nPages);
  } else {
    dprintf(2, "[INFO]: Parent: Allocating %ld Pages\n", nPages);
  }

  for(u_int64_t j = 0; j < nPages; j++) {
    char *ptr = mmap(NULL, sysconf(_SC_PAGESIZE), PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
    if(ptr == MAP_FAILED) {
      dprintf(2, "Unable to map memory. Error: %s\n", strerror(errno));
      return EXIT_FAILURE;
    }

    hvas[j] = (int64_t)(ptr);
    *(char *)(hvas[j]) = 0x2a;
  }


  addrInfo **aInfo = getAddrInfoFromHva(hvas, nPages);
  free(hvas);
  updatePhysicalInformation(aInfo, nPages);

  // Randomize
  aInfo = shuffleAInfoArray(aInfo, nPages);

  u_int64_t nSections = 0;
  char **names = getSectionNames(&nSections);

  if(!afterExec) {
    unlink("parent.log");
    unlink("parent_mapping.log");
    exportPages("parent_mapping.log", "parent", aInfo, nPages);
  } else {
    unlink("child.log");
    unlink("child_mapping.log");
    exportPages("child_mapping.log", "child", aInfo, nPages);
  }

  for(u_int64_t i = 0; i < nSections; i++) {
    u_int64_t nMappedPages = 0;
    u_int64_t *mappedPages = getMappedPages(&nMappedPages, names[i]);

    addrInfo **sectionInfo = getAddrInfoFromHva((int64_t *)mappedPages, nMappedPages);
    updatePhysicalInformation(sectionInfo, nMappedPages);

    char *name = names[i];
    if(strlen(name) == 0) {
      name = "<anonymous>";
    }

    if(!afterExec) {
      exportPages("parent.log", name, sectionInfo, nMappedPages);
    } else {
      exportPages("child.log", name, sectionInfo, nMappedPages);
    }
  }

  if(!afterExec) {
    // Allocate some pages
    void *p = mmap(NULL, nPagesRealloc * sysconf(_SC_PAGESIZE), PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
    for(u_int64_t j = 0; j < nPagesRealloc; j++) {
      *(volatile char *)(p + j * sysconf(_SC_PAGESIZE)) = 0x2a;
    }

    // Free some pages
    for(u_int64_t j = 0; j < + nFreePages && j < nPages; j++) {
      if(munmap((void *)(aInfo[j]->hva), sysconf(_SC_PAGESIZE)) != 0) {
        dprintf(2, "Unable to munmap page at offset %ld. Error: %s\n", j, strerror(errno));
      }
    }

    pid_t pid = fork();
    if(pid == 0) {
      execl("bin/measureRemap", "bin/measureRemap", "-a", childParamStr, NULL);
    } else if(pid == -1) {
      dprintf(2, "Unable to fork. Error: %s\n", strerror(errno));
    }
    int status = 0;
    waitpid(pid, &status, 0);

    if(status != 0) {
      return EXIT_FAILURE;
    }
  }

  return EXIT_SUCCESS;
}
