#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<fcntl.h>
#include<errno.h>
#include<math.h>

#include "memlib/util.h"
#include "evaluateAll.h"

/*******************************************************************************
 * compareResultByParentOffset takes two pointers to result pointers and
 * compares them by their parent offset. This function can be used to sort.
 *
 * @param r1: Pointer to the first result pointer
 * @param r2: Pointer to the second result pointer
 * @return r1->parentOffset - r2->parentOffset
 ******************************************************************************/
int compareResultByParentOffset(const void *r1, const void *r2) {
  result *res1 = *(result **)(r1);
  result *res2 = *(result **)(r2);
  return res1->parentOffset - res2->parentOffset;
}

/*******************************************************************************
 * constructResult allocates a new result structure, initializes it, and
 * returns a pointer to it.
 *
 * @param parentOffset: parent offset value that should be initialized in the
 *        newly allocated struct
 * @return Pointer to the new struct
 ******************************************************************************/
result *constructResult(u_int64_t parentOffset) {
  result *res = malloc(sizeof(result));
  res->parentOffset = parentOffset;
  res->childOffsets = NULL;
  res->nChildOffsets = 0;
  res->average = 0.0;
  res->variance = 0.0;
  res->standardDeviation = 0.0;
  return res;
}

/*******************************************************************************
 * addChildOffsetToResult takes a pointer to a result struct and a child offset
 * and adds the child offset to the submitted struct.
 *
 * @param res: Pointer to the result struct
 * @param childOffset: Offset that should be added
 * @return Pointer to the result struct that was modified
 ******************************************************************************/
result *addChildOffsetToResult(result *res, u_int64_t childOffset) {
  res->nChildOffsets += 1;
  res->childOffsets = realloc(res->childOffsets, sizeof(u_int64_t) * res->nChildOffsets);
  res->childOffsets[res->nChildOffsets - 1] = childOffset;
  return res;
}

/*******************************************************************************
 * calculateResultStats takes a pointer to a result struct and calculates
 * statistical values (currently, the average, variance and standard deviation).
 *
 * @param res: Pointer to the struct for which the statistics should be
 *        calculated
 * @return Pointer to the struct with added statistics
 ******************************************************************************/
result *calculateResultStats(result *res) {
  u_int64_t sum = 0;
  for(u_int64_t i = 0; i < res->nChildOffsets; i++) {
    sum += res->childOffsets[i];
  }
  res->average = 1.0 * sum / res->nChildOffsets;

  double varSum = 0.0;
  for(u_int64_t i = 0; i < res->nChildOffsets; i++) {
    varSum += (res->average - res->childOffsets[i]) * (res->average - res->childOffsets[i]);
  }
  res->variance = varSum / res->nChildOffsets;

  res->standardDeviation = sqrt(res->variance);

  return res;
}

/*******************************************************************************
 * constuctResults allocates and initializes a results struct and returns a
 * pointer to it.
 *
 * @return Pointer to the created struct
 ******************************************************************************/
results *constructResults() {
  results *res = malloc(sizeof(results));
  res->res = NULL;
  res->nRes = 0;
  return res;
}

/*******************************************************************************
 * addResultToResults takes a pointer to a results struct and a pointer to a
 * result that should be added and adds it to the results struct.
 *
 * @param res: Pointer to the results struct to which the new item should be
 *        added
 * @param r: Pointer that should be added to the results struct
 * @return Pointer to the modified results struct
 ******************************************************************************/
results *addResultToResults(results *res, result *r) {
  res->nRes += 1;
  res->res = realloc(res->res, sizeof(result) * res->nRes);
  res->res[res->nRes-1] = r;
  return res;
}

/*******************************************************************************
 * handleLine takes a line generated by bin/evaluate (that contains a parent and
 * child offset delimited by a space) and adds it to the submitted results
 * structure.
 *
 * @param res: Results structure the values from the line should be added to
 * @param line: String in the format described above
 * @return Modified results structure
 ******************************************************************************/
results *handleLine(results *res, char *line) {
  u_int64_t lineLen = strlen(line);
  char *parentStr = line;
  char *childStr = NULL;
  for(u_int64_t i = 0; i < lineLen; i++) {
    if(line[i] == ' ') {
      line[i] = 0;
      childStr = line + i + 1;
    }
  }
  if(childStr == NULL) {
    printf("[WARNING]: Unable to find child offset in line %s.\n", line);
    return res;
  }

  u_int64_t parentOffset = atoi(parentStr);
  u_int64_t childOffset = atoi(childStr);
  int match = 0;

  for(u_int64_t i = 0; i < res->nRes; i++) {
    if(res->res[i]->parentOffset == parentOffset) {
      match = 1;
      res->res[i] = addChildOffsetToResult(res->res[i], childOffset);
    }
  }

  if(!match) {
    result *r = constructResult(parentOffset);
    r = addChildOffsetToResult(r, childOffset);
    res = addResultToResults(res, r);
  }

  return res;
}

/*******************************************************************************
 * handleFile opens the submitted path and adds all lines to the submitted
 * results struct.
 *
 * @param res: Result stuct the content of the file should be added to
 * @param path: Path to the file that should be parsed
 * @return Modified results struct
 ******************************************************************************/
results *handleFile(results *res, char *path) {
  char *buf = readFile(path);
  char *line = strtok(buf, "\n");

  while(line != NULL) {
    res = handleLine(res, line);
    line = strtok(NULL, "\n");
  }

  return res;
}

/*******************************************************************************
 * calculateResultsStats takes a pointer to a results structs and calls
 * calculateResultStats for each result within the struct.
 *
 * @param res: Pointer to the struct which should be modified
 * @return Pointer to the modified results struct
 ******************************************************************************/
results *calculateResultsStats(results *res) {
  for(u_int64_t i = 0; i < res->nRes; i++) {
    res->res[i] = calculateResultStats(res->res[i]);
  }
  return res;
}

/*******************************************************************************
 * storeResults takes a pointer to a results struct, the path of a file the
 * results should be exported to, and a step width (e.g. how many data points
 * should be concluded to one line in the results file). It stores the results
 * from the struct to the file.
 *
 * @param res: Pointer to the results struct that contains the values that
 *        should be exported
 * @param path: Path of the file the exported data should be written to
 * @param stepWidth: Number of data points that should be concluded to one
 *        line in the results file (conclusion is done by calculating average
 *        values of all exported params)
 ******************************************************************************/
void storeResults(results *res, char *path, u_int64_t stepWidth) {
  int fd = open(path, O_WRONLY|O_CREAT|O_TRUNC, 0644);
  if(fd < 0) {
    printf("[ERROR]: Unable to open %s. Erorr: %s", path, strerror(errno));
    return;
  }

  for(u_int64_t i = 0; i < res->nRes; i+=stepWidth) {
    double parentOffsetSum = 0.0;
    double lowerValueSum = 0.0;
    double avgValueSum = 0.0;
    double upperValueSum = 0.0;
    for(u_int64_t j = 0; j < stepWidth; j++) {
      parentOffsetSum += res->res[i]->parentOffset;
      lowerValueSum += res->res[i]->average - res->res[i]->standardDeviation;
      avgValueSum += res->res[i]->average;
      upperValueSum += res->res[i]->average + res->res[i]->standardDeviation;
    }
    dprintf(fd, "%f %f %f %f\n", parentOffsetSum/stepWidth, lowerValueSum/stepWidth,avgValueSum/stepWidth,upperValueSum/stepWidth);
  }

  close(fd);
}

int main(int argc, char *argv[]) {
  if(argc < 2) {
    printf("Usage: %s file1 file2 ...\n", argv[0]);
    exit(EXIT_FAILURE);
  }

  results *res = constructResults();
  for(u_int64_t i = 1; i < argc; i++) {
    res = handleFile(res, argv[i]);
  }

  qsort(res->res, res->nRes, sizeof(result *), compareResultByParentOffset);

  res = calculateResultsStats(res);
  storeResults(res, "results.dat", 32);
}
