#include<stdlib.h>
#include<stdio.h>
#include<string.h>
#include<errno.h>
#include<getopt.h>
#include<unistd.h>
#include<fcntl.h>

#include "memlib/util.h"
#include "evaluate.h"

/*******************************************************************************
 * parseHex takes a hex string (starting with '0x') and parses it to an unsigned
 * integer.
 *
 * @param str: String representation of the hex value
 * @return Value represented by str
 ******************************************************************************/
u_int64_t parseHex(char *str) {
  u_int64_t hex = 0;
  if(strlen(str) < 3 || str[0] != '0' || str[1] != 'x') {
    dprintf(2, "[WARNING]: Unable to parse %s as hex\n.", str);
    return 0;
  }

  for(u_int64_t i = 2; i < strlen(str); i++) {
    hex *= 16;
    if (str[i] >= '0' && str[i] <= '9') {
      hex += str[i] - '0';
    } else if (str[i] >= 'a' && str[i] <= 'f') {
      hex += str[i] - 'a' + 10;
    } else if (str[i] >= 'A' && str[i] <= 'F') {
      hex += str[i] - 'A' + 10;
    } else {
      dprintf(2, "[WARNING]: Unable to parse %s as hex ('%c' is invalid).\n.", str, str[i]);
      return 0;
    }
  }

  return hex;
}

/*******************************************************************************
 * getPfnList takes a list of strings and the number of items in the list and
 * generates a pfn list from them.
 *
 * @param pfns: List that contains the strings that should be added to the pfn
 *        list
 * @param nPfns: Number of items in pfns
 * @return Pointer to a pfnList struct that stores the submitted values.
 ******************************************************************************/
pfnList *getPfnList(char **pfns, u_int64_t nPfns) {
  pfnList *list = malloc(sizeof(pfnList));
  list->pfns = malloc(sizeof(u_int64_t) * nPfns);
  list->nPfns = nPfns;

  for(u_int64_t i = 0; i < nPfns; i++) {
    list->pfns[i] = parseHex(pfns[i]);
  }

  return list;
}

/*******************************************************************************
 * getMapSection takes a pfnList and a section name and generates a map section
 * struct from it.
 *
 * @param pfns: Pointer to the pfnList struct that should be added
 * @param sectionName: Name of the section
 * @return Pointer to the newly created mapSection struct
 ******************************************************************************/
mapSection *getMapSection(pfnList *pfns, char *sectionName) {
  mapSection *section = malloc(sizeof(mapSection));
  section->pfns = pfns;
  section->name = sectionName;

  return section;
}

/*******************************************************************************
 * constructMapSections generates an empty mapSections structure
 *
 * @return Pointer to the newly allocated and initialized structure
 ******************************************************************************/
mapSections *constructMapSections() {
  mapSections *sections = malloc(sizeof(mapSections));
  sections->sections = NULL;
  sections->nSections = 0;

  return sections;
}

/*******************************************************************************
 * addToMapSections takes a pointer to a mapSection struct that should be added
 * to the mapSections struct.
 *
 * @param sections: Pointer to the mapSections struct the new element should be
 *        added to
 * @param section: Pointer to a mapSection struct that should be added
 * @return Pointer to the modified mapSections struct
 ******************************************************************************/
mapSections *addToMapSections(mapSections *sections, mapSection *section) {
  sections->nSections += 1;
  sections->sections = realloc(sections->sections, sizeof(mapSection *) * sections->nSections);
  sections->sections[sections->nSections-1] = section;
  return sections;
}

/*******************************************************************************
 * parseMapFile loads a map file exported by the parent or child and parses it
 * into a mapSections struct
 *
 * @param mapFile: Path to the file that should be parsed
 * @return Pointer to the mapSections struct that contains the parsed data
 ******************************************************************************/
mapSections *parseMapFile(char *mapFile) {
  mapSections *sections = constructMapSections();

  char *buf = readFile(mapFile);
  char *sectionName = "";
  char **pfns = NULL;
  u_int64_t nPfns = 0;

  char *line = strtok(buf, "\n");

  while(line != NULL) {
    if(strlen(line) >= 3 && line[0] == '0' && line[1] == 'x') {
      nPfns += 1;
      pfns = realloc(pfns, sizeof(char *) * nPfns);
      pfns[nPfns-1] = line;
    } else {
      if(nPfns > 0) {
        pfnList *pList = getPfnList(pfns, nPfns);
        mapSection *section = getMapSection(pList, sectionName);
        sections = addToMapSections(sections, section);
      }
      nPfns = 0;
      free(pfns);
      pfns = NULL;
      sectionName = line;
    }

    line = strtok(NULL, "\n");
  }

  if(nPfns > 0) {
    pfnList *pList = getPfnList(pfns, nPfns);
    mapSection *section = getMapSection(pList, sectionName);
    sections = addToMapSections(sections, section);
  }
  nPfns = 0;
  free(pfns);
  pfns = NULL;

  return sections;
}

/*******************************************************************************
 * dumpMapSections dumps the content of a mapSections struct, used for debugging
 *
 * @param sections: Pointer to the struct that should be dumped
 ******************************************************************************/
void dumpMapSections(mapSections *sections) {
  for(u_int64_t i = 0; i < sections->nSections; i++) {
    printf("Section '%s'\n", sections->sections[i]->name);
    for(u_int64_t j = 0; j < sections->sections[i]->pfns->nPfns; j++) {
      printf("0x%lx\n", sections->sections[i]->pfns->pfns[j]);
    }
  }
}

/*******************************************************************************
 * getSectionAndOffset searches a mapSections struct for a submitted PFN and
 * returns the name of the section and the offset within that section.
 *
 * @param sections: mapSections struct that should be used for search
 * @param searchPfn: Pfn that is searched
 * @param offset: Pointer to store the offset if a match was found
 * @return Name of the section the match was found in
 ******************************************************************************/
char *getSectionAndOffset(mapSections *sections, u_int64_t searchPfn, u_int64_t *offset) {
  for(u_int64_t i = 0; i < sections->nSections; i++) {
    for(u_int64_t j = 0; j < sections->sections[i]->pfns->nPfns; j++) {
      if(searchPfn == sections->sections[i]->pfns->pfns[j]) {
        *offset = j;
        return sections->sections[i]->name;
      }
    }
  }
  *offset = 0;
  return NULL;
}

/*******************************************************************************
 * printHelp prints the usage page of the program and exits using a submitted
 * exit code
 *
 * @param binary: Name of the executable, e.g. argv[0]
 * @param exitCode: Code that should be used to exit
 ******************************************************************************/
void printHelp(char *binary, int exitCode) {
  printf("Usage: %s -p parentFile -c childFile [-e exportDir] [-d] [-h]\n", binary);
  printf("\t-p: Path to the file that contains the PFNs of the mappings of the parent.\n");
  printf("\t-c: Path to the file that contains the sections and PFNs of the child\n");
  printf("\t-e: Path to create the export files (e.g. directory)\n");
  printf("\t-d: Show debug information\n");
  printf("\t-h: Show this help message\n");
  exit(exitCode);
}

int main(int argc, char *argv[]) {
  char *parentFile = NULL;
  char *childFile = NULL;
  char *exportDir = NULL;
  int debug = 0;

  int opt;
  extern char *optarg;
  while((opt = getopt(argc, argv, "p:c:e:hd")) != -1) {
    switch(opt) {
      case 'p':
        parentFile = optarg;
        break;
      case 'c':
        childFile = optarg;
        break;
      case 'e':
        exportDir = optarg;
        break;
      case 'h':
        printHelp(argv[0], EXIT_SUCCESS);
        break;
      case 'd':
        debug = 1;
        break;
      default:
        dprintf(2, "Unknown option '%c'.\n", opt);
        printHelp(argv[0], EXIT_FAILURE);
        break;
    }
  }

  if(parentFile == NULL) {
    dprintf(2, "[ERROR]: No parent file specified (use '-p')\n");
    printHelp(argv[0], EXIT_FAILURE);
  }

  if(childFile == NULL) {
    dprintf(2, "[ERROR]: No child file specified (use '-c')\n");
    printHelp(argv[0], EXIT_FAILURE);
  }

  mapSections *parentSections = parseMapFile(parentFile);
  mapSections *childSections = parseMapFile(childFile);

  char *childSection = NULL;
  u_int64_t childOffset = 0;
  for(u_int64_t i = 0; i < parentSections->nSections; i++) {
    for(u_int64_t j = 0; j < parentSections->sections[i]->pfns->nPfns; j++) {
      childSection = getSectionAndOffset(childSections, parentSections->sections[i]->pfns->pfns[j], &childOffset);
      if(childSection == NULL) {
        continue;
      }
      if(debug) {
        printf("'%s'[%ld] -> '%s'[%ld]\n", parentSections->sections[i]->name, j, childSection, childOffset);
      }
      if(exportDir) {
        char *parentSection = parentSections->sections[i]->name;
        char *fileName = malloc(sizeof(char) * (strlen(parentSection) + strlen("_") + strlen(childSection) + strlen(".dat") + 1));
        sprintf(fileName, "%s_%s.dat", parentSection, childSection);
        for(u_int64_t z = 0; z < strlen(fileName); z++) {
          switch(fileName[z]) {
            case '<':
            case '>':
            case '\'':
            case '[':
            case ']':
            case '/':
              fileName[z] = '_';
          }
        }
        char *filePath = malloc(sizeof(char) * (strlen(exportDir) + strlen("/") + strlen(fileName) + 1));
        sprintf(filePath, "%s/%s", exportDir, fileName);

        int fd = open(filePath, O_WRONLY|O_CREAT|O_APPEND, 0644);
        if(fd < 0) {
          dprintf(2, "Unable to open %s. Error: %s\n", filePath, strerror(errno));
          return EXIT_FAILURE;
        }

        dprintf(fd, "%ld %ld\n", j, childOffset);

        close(fd);
      }
    }
  }
}
