#include<stdio.h>
#include<stdlib.h>
#include<getopt.h>
#include<string.h>
#include<sys/types.h>
#include<dirent.h>
#include<errno.h>
#include<sys/stat.h>
#include<sys/param.h>

#include "memlib/util.h"
#include "evaluate.h"

u_int64_t hexstringToLong(char *hex) {
  u_int64_t num = 0;

  if(strlen(hex) < 3) {
    return 0;
  }

  if(hex[0] != '0' || hex[1] != 'x') {
    return 0;
  }

  for(u_int64_t i = 2; hex[i] != 0; i++) {
    num *= 16;

    if(hex[i] >= '0' && hex[i] <= '9') {
      num += hex[i] - '0';
    }
    if(hex[i] >= 'a' && hex[i] <= 'f') {
      num += hex[i] - 'a' + 10;
    }
    if(hex[i] >= 'A' && hex[i] <= 'F') {
      num += hex[i] - 'A' + 10;
    }
  }

  return num;
}

void initBuddyInfoCache(buddyInfoCache *cache) {
  cache->buddyInfo = NULL;
  cache->nBuddyInfo = 0;
  cache->buddyLines = NULL;
  cache->nBuddyLines = 0;
  cache->oldBasePath = NULL;
}

void cacheBuddyInfoCallback(char *path, buddyInfoCache *cache) {
  char *buf = readFile(path);
  if(buf == NULL) {
    return;
  }

  u_int64_t pathSize = strlen(path);
  u_int64_t bufSize = strlen(buf);

  if(bufSize == 0) {
    return;
  }

  cache->nBuddyInfo += pathSize + 1 + bufSize;
  cache->buddyInfo = realloc(cache->buddyInfo, (cache->nBuddyInfo + 1) * sizeof(char));

  strncpy(cache->buddyInfo + cache->nBuddyInfo - bufSize - 1 - pathSize, path, pathSize + 1);
  strncpy(cache->buddyInfo + cache->nBuddyInfo - bufSize - 1, "\n", 2);
  strncpy(cache->buddyInfo + cache->nBuddyInfo - bufSize, buf, bufSize + 1);
  //cache->buddyInfo[cache->nBuddyInfo] = 0;

  //printf("Buddy info: %s\n", cache->buddyInfo);
  //exit(0);

  free(buf);
}

char **cacheBuddyInfo(char *basepath, u_int64_t *nLines, buddyInfoCache *cache) {
  if(cache->buddyInfo != NULL && strcmp(cache->oldBasePath, basepath) == 0) {
    *nLines = cache->nBuddyLines;
    return cache->buddyLines;
  }

  if(cache->oldBasePath == NULL || strcmp(cache->oldBasePath, basepath) != 0) {
    free(cache->oldBasePath);
    cache->oldBasePath = malloc((strlen(basepath) + 1) * sizeof(char));
    strncpy(cache->oldBasePath, basepath, strlen(basepath) + 1);
  }

  dirWalk(basepath, cacheBuddyInfoCallback, cache);

  char *line = strtok(cache->buddyInfo, "\n");
  while(line != NULL) {
    cache->nBuddyLines += 1;
    cache->buddyLines = realloc(cache->buddyLines, cache->nBuddyLines * sizeof(char *));
    cache->buddyLines[cache->nBuddyLines - 1] = line;
    line = strtok(NULL, "\n");
  }

  *nLines = cache->nBuddyLines;
  return cache->buddyLines;
}

char *checkBuddyInfo(char **buddyLines, u_int64_t nBuddyLines, char *pfn, u_int64_t *offset) {
  char *currentSection = NULL;
  *offset = 0;
  for(u_int64_t i = 0; i < nBuddyLines; i++) {
    char *line = buddyLines[i];
    if(line[0] != '0' && line[1] != 'x') {
      currentSection = line;
      *offset = 0;
      continue;
    }

    *offset += 1;

    if(strcmp(line, pfn) == 0) {
      return currentSection;
    }
  }

  return NULL;
}

void dirWalk(char *path, void(*buddyCheckerCallback)(char *, buddyInfoCache *), buddyInfoCache *cache) {
  char currentPath[MAXPATHLEN];
  sprintf(currentPath, path);

  DIR *dir = opendir(path);
  if(dir == NULL) {
    dprintf(2, "Unable to open dir %s. Error: %s\n", path, strerror(errno));
    return;
  }

  struct dirent *entry;
  struct stat entryStat;
  errno = 0;
  while((entry = readdir(dir)) ) {
    if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) {
			continue;
		}

    snprintf(currentPath, MAXPATHLEN, "%s/%s", path, entry->d_name);

    if(stat(currentPath, &entryStat) != 0) {
      dprintf(2, "Unable to stat %s. Error: %s\n", currentPath, strerror(errno));
      continue;
    }

    if((entryStat.st_mode & S_IFMT) == S_IFDIR) {
      dirWalk(currentPath, buddyCheckerCallback, cache);
    } else if((entryStat.st_mode & S_IFMT) == S_IFREG) {
      buddyCheckerCallback(currentPath, cache);
    }

  }

  if(errno != 0) {
    dprintf(2, "Unable to read content of dir %s. Error: %s\n", currentPath, strerror(errno));
  }

  if(closedir(dir) != 0) {
    dprintf(2, "Unable to close dir %s. Error: %s\n", currentPath, strerror(errno));
  }
}

char *getBuddyInfo(char *basepath, char *pfn, u_int64_t *offset, buddyInfoCache *cache) {
  u_int64_t nLines = 0;
  char **buddyLines = cacheBuddyInfo(basepath, &nLines, cache);
  return checkBuddyInfo(buddyLines, nLines, pfn, offset);
}

char *getPfnFromTraceEntryLine(char *line) {
  u_int64_t lineOffset = 0;
  char *searchStr = "pfn=";
  u_int64_t lineLength = strlen(line);
  u_int64_t searchStrLen = strlen(searchStr);
  for(lineOffset = 0; (int64_t)(lineOffset) < (int64_t)(lineLength - searchStrLen) && strncmp(line + lineOffset, searchStr, searchStrLen) !=0; lineOffset++) {
    //count until the substring matches "pfn="
  }

  lineOffset += strlen(searchStr);

  char *pfn = NULL;
  u_int64_t len = 0;
  while(line[lineOffset] != ' ') {
    len++;
    pfn=realloc(pfn, (len + 1) * sizeof(char));
    pfn[len-1] = line[lineOffset];
    pfn[len] = 0;
    lineOffset++;
  }

  return pfn;
}

int compareTraceEntriesByPfn(const void *e1, const void *e2) {
  char *s1 = *(char **)(e1);
  char *s2 = *(char **)(e2);

  char *pfn1Str = getPfnFromTraceEntryLine(s1);
  char *pfn2Str = getPfnFromTraceEntryLine(s2);

  u_int64_t pfn1 = hexstringToLong(pfn1Str);
  u_int64_t pfn2 = hexstringToLong(pfn2Str);

  free(pfn1Str);
  free(pfn2Str);

  return pfn1 - pfn2;
}

int64_t searchLinesInPfnEntries(char *pfn, char **lines, u_int64_t nLines, u_int64_t *nMatches) {
  u_int64_t searchPfn = hexstringToLong(pfn);

  u_int64_t lower = 0;
  u_int64_t upper = nLines;

  u_int64_t idx = 0;
  u_int64_t lastIdx = idx;

  *nMatches = 0;

  while(1) {
    lastIdx = idx;
    idx = (lower + upper) / 2;
    if(lastIdx == idx) {
      break;
    }
    char *linePfnStr = getPfnFromTraceEntryLine(lines[idx]);
    u_int64_t linePfn = hexstringToLong(linePfnStr);
    free(linePfnStr);

    if(searchPfn == linePfn) {
      while(1) {
        idx--;
        char *linePfnStr = getPfnFromTraceEntryLine(lines[idx]);
        u_int64_t linePfn = hexstringToLong(linePfnStr);
        free(linePfnStr);

        if(linePfn != searchPfn) {
          idx++;
          break;
        }

        if(idx == 0) {
          break;
        }
      }

      while(1) {
        *nMatches += 1;
        if(idx + *nMatches == nLines) {
          break;
        }

        char *linePfnStr = getPfnFromTraceEntryLine(lines[idx + *nMatches]);
        u_int64_t linePfn = hexstringToLong(linePfnStr);
        free(linePfnStr);

        if(linePfn != searchPfn) {
          break;
        }
      }
      return idx;
    }

    if(lower == upper) {
      break;
    }

    if(searchPfn > linePfn) {
      lower=idx;
    } else {
      upper=idx;
    }
    if(idx == nLines - 1) {
      break;
    }
  }
  return -1;
}

void printHelp(char *binary, int exitCode) {
  printf("Usage: %s [-p parent.log] [-c child.log] [-t trace.txt] [-h]\n", binary);
  printf("\t-p: Specify the log file created by the parent process (default 'parent.log')\n");
  printf("\t-c: Specify the log file created by the child process (default 'child.log')\n");
  printf("\t-t: Specify the tracing log file created by trace-cmd (default 'trace.log', use 'none' to disable tracing log file)\n");
  printf("\t-h: Show this help message\n");

  exit(exitCode);
}

char **getLinesFromBuf(char *buf, u_int64_t *len) {
  char **lines = NULL;
  *len = 0;

  char *line = strtok(buf, "\n");
  while(line != NULL) {
    *len += 1;
    lines = realloc(lines, (*len) * sizeof(char *));
    lines[(*len) - 1] = line;

    line = strtok(NULL, "\n");
  }

  return lines;
}

int main(int argc, char *const argv[]) {
  char *parentFile = "parent.log";
  char *childFile = "child.log";
  char *traceFile = "trace.log";

  int opt;
  extern char *optarg;
  while((opt = getopt(argc, argv, "p:c:t:h")) != -1) {
    switch(opt) {
      case 'p':
        parentFile = optarg;
        break;
      case 'c':
        childFile = optarg;
        break;
      case 't':
        traceFile = optarg;
        break;
      case 'h':
        printHelp(argv[0], EXIT_SUCCESS);
        break;
      default:
        printHelp(argv[0], EXIT_FAILURE);
        break;
    }
  }

  dprintf(2, "Evaluation process started.\n");

  char *parentBuf = readFile(parentFile);
  char *childBuf = readFile(childFile);
  char *traceBuf = NULL;
  
  if(strlen(traceFile) != 4 || strncmp(traceFile, "none", 4) != 0) {
    traceBuf = readFile(traceFile);
  }

  u_int64_t nParentLines = 0;
  char **parentLines = getLinesFromBuf(parentBuf, &nParentLines);

  u_int64_t nChildLines = 0;
  char **childLines = getLinesFromBuf(childBuf, &nChildLines);
  u_int8_t *newChildLines = malloc(sizeof(u_int8_t) * nChildLines);
  for(u_int64_t i = 0; i < nChildLines; i++) {
    newChildLines[i] = 0;
  }

  u_int64_t nTraceLines = 0;
  char **traceLines = NULL;
  if(traceBuf != NULL) {
    traceLines = getLinesFromBuf(traceBuf, &nTraceLines);
    qsort(traceLines, nTraceLines, sizeof(char *), compareTraceEntriesByPfn);
  }

  char *parentSection = "";
  char *childSection = "";
  u_int64_t parentSectionOffset = 0;
  u_int64_t childSectionOffset = 0;

  buddyInfoCache beforeCache;
  buddyInfoCache betweenCache;
  buddyInfoCache afterCache;
  initBuddyInfoCache(&beforeCache);
  initBuddyInfoCache(&betweenCache);
  initBuddyInfoCache(&afterCache);

  for(u_int64_t parentOffset = 0; parentOffset < nParentLines; parentOffset++) {
    u_int64_t percBaseValue = parentOffset * 10000 / nParentLines;
    dprintf(2, "\rProcessing parent page %ld of %ld (%3ld.%02ld%%)", parentOffset, nParentLines, percBaseValue / 100, percBaseValue % 100);
    fflush(stderr);
    char *parentLine = parentLines[parentOffset];

    //If the line does not start with "0x", it is a section name
    if(parentLine[0] != '0' || parentLine[1] != 'x') {
      parentSection = parentLine;
      parentSectionOffset = 0;
      continue;
    }

    parentSectionOffset++;

    u_int64_t reallocated = 0;
    childSectionOffset = 0;
    for(u_int64_t childOffset = 0; childOffset < nChildLines; childOffset++) {
      char *childLine = childLines[childOffset];

      if(childLine[0] != '0' || childLine[1] != 'x') {
        childSection = childLine;
        childSectionOffset = 0;
        newChildLines[childOffset] = 0;
        continue;
      }

      childSectionOffset++;

      int len = strlen(parentLine);
      if(strlen(childLine) != len) {
        continue;
      }

      if(strncmp(parentLine, childLine, len) != 0) {
        continue;
      }

      printf("PFN %s was reallocated (section '%s' -> '%s') [offset %ld -> %ld]\n", parentLine, parentSection, childSection, parentSectionOffset, childSectionOffset);
      newChildLines[childOffset] = 1;
      reallocated = 1;
    }

    if(reallocated == 0) {
      printf("PFN %s was not reallocated (section '%s' -> ?) [offset %ld -> ?]\n", parentLine, parentSection, parentSectionOffset);
    }

    u_int64_t buddyOffset = 0;

    char *buddyInfoBefore = getBuddyInfo("data/allocTraceBefore", parentLine, &buddyOffset, &beforeCache);
    if(buddyInfoBefore != NULL) {
      printf("\tBuddy matches before execution: %s at offset %ld\n", buddyInfoBefore, buddyOffset);
    }

    char *buddyInfoBetween = getBuddyInfo("data/allocTraceBetween", parentLine, &buddyOffset, &betweenCache);
    if(buddyInfoBetween != NULL) {
      printf("\tBuddy matches between execution: %s at offset %ld\n", buddyInfoBetween, buddyOffset);
    }

    char *buddyInfoAfter = getBuddyInfo("data/allocTraceAfter", parentLine, &buddyOffset, &afterCache);
    if(buddyInfoAfter != NULL) {
      printf("\tBuddy matches after execution: %s at offset %ld\n", buddyInfoAfter, buddyOffset);
    }

    u_int64_t nTraceMatches = 0;
    int64_t traceOffset = searchLinesInPfnEntries(parentLine, traceLines, nTraceLines, &nTraceMatches);
    if(traceOffset < 0) {
      continue;
    }

    for(u_int64_t traceLineOffset = 0; traceLineOffset < nTraceMatches; traceLineOffset++) {
      printf("%s\n", traceLines[traceOffset + traceLineOffset]);
    }
  }

  childSection = "<unknown>";
  childSectionOffset = 0;
  for(u_int64_t childOffset = 0; childOffset < nChildLines; childOffset++) {
    char *childLine = childLines[childOffset];

    if(childLine[0] != '0' || childLine[1] != 'x') {
      childSection = childLine;
      childSectionOffset = 0;
      continue;
    }

    childSectionOffset++;

    if(newChildLines[childOffset] == 1) {
      // Line was reallocated and already print as part of the parent section
      continue;
    }

    printf("PFN %s was not reallocated (section ? -> '%s') [offset ? -> %ld]\n", childLine, childSection, childSectionOffset);

    u_int64_t nTraceMatches = 0;
    int64_t traceOffset = searchLinesInPfnEntries(childLine, traceLines, nTraceLines, &nTraceMatches);
    if(traceOffset < 0) {
      continue;
    }

    for(u_int64_t traceLineOffset = 0; traceLineOffset < nTraceMatches; traceLineOffset++) {
      printf("%s\n", traceLines[traceOffset + traceLineOffset]);
    }
  }

  dprintf(2, "\n");
}
