#include<stdio.h>
#include<stdint.h>
#include<stdlib.h>

#include "measure.h"
#include "asm.h"
#include "helper.h"

/*******************************************************************************
 * measure_time takes two addresses and alternatingly accesses them for
 * N_TIME_MEASUREMENTS times. During that, the time is measured using
 * rdtscp.
 *
 * @param a1: First address
 * @param a2: Second address
 * @return Time the alternating access took in average over N_TIME_MEASUREMENTS
 *         measurements.
 ******************************************************************************/
u_int64_t measure_time(volatile char *a1, volatile char *a2) {
    uint64_t before, after;
    before = rdtscp();
    lfence();
    for (u_int64_t i = 0; i < N_TIME_MEASUREMENTS; i++) {
        (void)*a1;
        (void)*a2;
        clflushopt(a1);
        clflushopt(a2);
        mfence();
    }
    after = rdtscp();
    return (u_int64_t) ((after - before)/N_TIME_MEASUREMENTS);
}

/*******************************************************************************
 * measure_single_threshold allocates one THP and measures the threshold value
 * for that THP.
 *
 * @param debug: If set to 1, additional debug information are shown
 * @return Threshold between row hit and row conflict
 ******************************************************************************/
u_int64_t measure_single_threshold(u_int64_t debug) {
    addressGroup *hugePages = getHugepages(1);

    u_int64_t *times = malloc(N_PAGES_FOR_THP * sizeof(u_int64_t));
    for(u_int64_t i = 0; i < N_PAGES_FOR_THP; i++) {
        times[i] = measure_time(hugePages->addresses[0], hugePages->addresses[0] + i * getPageSize());
    }

    qsort(times, N_PAGES_FOR_THP, sizeof(u_int64_t), compareInt);

    u_int64_t offset = times[0] / 10;
    u_int64_t size = times[N_PAGES_FOR_THP - 1] / 10 - offset + 1;

    u_int64_t *accesses = malloc(size * sizeof(u_int64_t));

    for(u_int64_t i = 0; i < size; i++) {
        accesses[i] = 0;
    }

    for(u_int64_t i = 0; i < N_PAGES_FOR_THP; i++) {
        accesses[times[i]/10 - offset] += 1;
    }

    u_int64_t lastValue = 0;
    u_int64_t retVal = -1;
    u_int64_t foundMatches = 0;
    for(u_int64_t i = 0; i < size; i++) {
        foundMatches += accesses[i];
        if(accesses[i] > 2) {
            // There has to be at least 2 (because grouping by 10, it is equal
            // to 20) empty values. Also, at least 3/4 of all timings has to be
            // before the match (for 8 banks, 1/8 can be expected to be slow,
            // for 16 banks 1/16).
            if (lastValue < i - 2 && foundMatches >= (N_PAGES_FOR_THP * 3) / 4) {
                u_int64_t candidate = (((i + lastValue) / 2) + offset) * 10;
                if(debug) {
                    printf("Found candidate: %ld\n", candidate);
                }

                if(retVal == -1) {
                    retVal = candidate;
                }

                if(!debug) {
                    break;
                }
            }
            lastValue = i;
        }
        if(debug) {
            printf("%4ld: %.*s\n", (i + offset) * 10, (int)(accesses[i]), "################################################################################################################################################################################################");
        }
    }

    freeAddressGroup(hugePages, 1);
    free(times);
    free(accesses);

    return retVal;

}

/*******************************************************************************
 * measure_system_threshold utilizes measure_single_threshold() to get a more
 * accurate threshold by measuring multiple times. Afterwards, the median is
 * used as result.
 *
 * @param debug: If set to 1, additional debug information are shown
 * @param nMeasurements: Number of measurements
 * @return median of nMeasurements threshold measurements
 ******************************************************************************/
u_int64_t measure_system_threshold(u_int64_t debug, u_int64_t nMeasurements) {
    u_int64_t *thresholds = malloc(nMeasurements * sizeof(u_int64_t));
    for(u_int64_t i = 0; i < nMeasurements; i++) {
        thresholds[i] = measure_single_threshold(debug);
    }
    qsort(thresholds, nMeasurements, sizeof(u_int64_t), compareInt);

    if(debug) {
        for(u_int64_t i = 0; i < nMeasurements; i++) {
            printf("thresholds[%ld] = %ld\n", i, thresholds[i]);
        }
    }

    u_int64_t threshold = thresholds[nMeasurements/2];
    free(thresholds);
    return threshold;
}
