#include<stdio.h>
#include<stdlib.h>
#include<sys/mman.h>
#include<malloc.h>
#include<unistd.h>
#include<math.h>
#include<fcntl.h>
#include<errno.h>
#include<string.h>

#include "helper.h"
#include "memlib/memoryInspect.h"

/*******************************************************************************
 * compareInt is a helper function for qsort()
 *
 * @param v1: Pointer to the first int
 * @param v2: Pointer to the second int
 * @return *v1 - *v2
 ******************************************************************************/
int compareInt(const void *v1, const void *v2) {
    return (int)(*(u_int64_t*)(v1) - *(u_int64_t*)(v2));
}

/*******************************************************************************
 * getRandomIntWithinBounds takes a minimum and maximum bound and returns a
 * random int between those bounds
 *
 * @param min: Lower boundary
 * @param max: Upper boundary
 * @return Random int between min and max, including min and excluding max
 ******************************************************************************/
u_int64_t getRandomIntWithinBounds(u_int64_t min, u_int64_t max) {
    return rand() % (max - min) + min;
}

/*******************************************************************************
 * shuffleUInt64Array takes an u_int64_t array and shuffles the elements.
 *
 * @param arr: Array that should be shuffled
 * @param len: Number of the elements in arr
 * @return shuffled arr
 ******************************************************************************/
u_int64_t *shuffleUInt64Array(u_int64_t *arr, u_int64_t len) {
    u_int64_t j;
    for(u_int64_t i = 0; i < len - 1; i++) {
        j = getRandomIntWithinBounds(i, len);
        arr[i] ^= arr[j];
        arr[j] ^= arr[i];
        arr[i] ^= arr[j];
    }

    return arr;
}

/*******************************************************************************
 * getHugepages takes a number and allocates the according amount of huge pages.
 *
 * @param nHugepages: Number of 2M THP that should be allocated
 * @return addressGroup containing the allocated hugepages
 ******************************************************************************/
addressGroup *getHugepages(u_int64_t nHugepages) {
    addressGroup *aGroup = initializeAddressGroup();
    u_int64_t pageSize = getPageSize();

    for(u_int64_t i = 0; i < nHugepages; i++) {
        volatile char *hugepage = memalign(pageSize * N_PAGES_FOR_THP, N_PAGES_FOR_THP * pageSize * sizeof(char));
        madvise((char *)hugepage, N_PAGES_FOR_THP * pageSize * sizeof(char), MADV_HUGEPAGE);

        for(u_int64_t i = 0; i < 512; i++) {
            hugepage[i * pageSize] = 0x42;
        }

        addAddressToAddressGroup(aGroup, hugepage, pageSize * 512);
    }

    return aGroup;
}

/*******************************************************************************
 * getPageSize is a helper function that retrieves and returns the page size
 *
 * @return page size on the current system
 ******************************************************************************/
u_int64_t getPageSize() {
    return sysconf(_SC_PAGESIZE);
}

/*******************************************************************************
 * initializeAddressGroup constructs a new addressGroup struct
 *
 * @return Pointer to the newly created address group
 ******************************************************************************/
addressGroup *initializeAddressGroup() {
    addressGroup *aGroup = malloc(sizeof(addressGroup));
    aGroup->addresses = NULL;
    aGroup->addressSizes = NULL;
    aGroup->nAddresses = 0;

    return aGroup;
}

/*******************************************************************************
 * freeAddressGroup takes a pointer to an existing address group and destructs
 * it
 *
 * @param aGroup: Pointer to the address group that should be destructed
 * @param recursive: If set, the addresses within the address group are freed
 *        as well. Otherwise, addresses within the struct are not modified.
 * @return: NULL is returned.
 ******************************************************************************/
addressGroup *freeAddressGroup(addressGroup *aGroup, u_int64_t recursive) {
    if(recursive) {
        for(u_int64_t i = 0; i < aGroup->nAddresses; i++) {
            free((char*)aGroup->addresses[i]);
            aGroup->addresses[i] = NULL;
        }
        aGroup->nAddresses = 0;
    }

    free(aGroup->addresses);
    free(aGroup->addressSizes);
    free(aGroup);

    return NULL;
}

/*******************************************************************************
 * addAddressToAddressGroup takes a pointer to an address group and the address
 * that should be added and adds it.
 *
 * @param aGroup: Address group the new address should be added to
 * @param address: Address that should be added to the address group
 * @param addressSize: Size of the buffer the address points to (in bytes)
 * @return modified address group
 ******************************************************************************/
addressGroup *addAddressToAddressGroup(addressGroup *aGroup, volatile char *address, u_int64_t addressSize) {
    aGroup->nAddresses++;

    aGroup->addresses = realloc(aGroup->addresses, aGroup->nAddresses * sizeof(volatile char *));
    aGroup->addressSizes = realloc(aGroup->addressSizes, aGroup->nAddresses * sizeof(u_int64_t));

    aGroup->addresses[aGroup->nAddresses - 1] = address;
    aGroup->addressSizes[aGroup->nAddresses - 1] = addressSize;

    return aGroup;
}

/*******************************************************************************
 * removeAddressFromAddressGroup takes an address group and the index of the
 * address that should be removed and removes the address from the address
 * group.
 *
 * @param aGroup: Address group the address should be removed from
 * @param index: Index of the address that should be removed
 * @param cleanupAddress: If set to 1, the address is freed. Otherwise, it is
 *        only removed from the address group and not modified.
 * @return modified address group
 ******************************************************************************/
addressGroup *removeAddressFromAddressGroup(addressGroup *aGroup, u_int64_t index, u_int64_t cleanupAddress) {
    aGroup->nAddresses--;

    if(cleanupAddress) {
        free((char*)aGroup->addresses[index]);
    }

    aGroup->addresses[index] = aGroup->addresses[aGroup->nAddresses];
    aGroup->addresses[aGroup->nAddresses] = NULL;
    aGroup->addressSizes[index] = aGroup->addressSizes[aGroup->nAddresses];
    aGroup->addressSizes[aGroup->nAddresses] = 0;

    return aGroup;
}

/*******************************************************************************
 * removeAddressGroupFromAddressGroupsWithoutRelocate removes an address from an
 * address group but does not relocate the data, but sets the corresponding
 * address to NULL.
 *
 * @param aGroup: Address group the address should be removed from
 * @param index: Index of the address that should be removed
 * @param cleanupAddress: If set to 1, the address is freed. Otherwise, it is
 *        only removed from the address group and not modified.
 * @return modified address group
 ******************************************************************************/
addressGroup *removeAddressFromAddressGroupWithoutRelocate(addressGroup *aGroup, u_int64_t index, u_int64_t cleanupAddress) {
    if(cleanupAddress) {
        free((char*)aGroup->addresses[index]);
    }

    aGroup->addresses[index] = NULL;
    aGroup->addressSizes[index] = 0;

    return aGroup;
}

/*******************************************************************************
 * cleanAddressGroup takes an address group and relocates all address that are
 * NULL.
 *
 * @param aGroup: Address group that should be cleaned.
 * @return modified address group
 ******************************************************************************/
addressGroup *cleanAddressGroup(addressGroup *aGroup) {
    for(u_int64_t i = 0; i < aGroup->nAddresses; i++) {
        if(aGroup->addresses[i] == NULL || aGroup->addressSizes[i] == 0) {
            aGroup = removeAddressFromAddressGroup(aGroup, i, 0);
            i--;
        }
    }

    return aGroup;
}

/*******************************************************************************
 * printAddressGroup takes an address group and dumps it to stdout.
 *
 * @param aGroup: Address group that should be printed
 ******************************************************************************/
void printAddressGroup(addressGroup *aGroup) {
    printf("Address Group at %p contains %ld addresses:\n", aGroup, aGroup->nAddresses);
    for(u_int64_t i = 0; i < aGroup->nAddresses; i++) {
        printf("\t%p (size: %ld bytes)\n", aGroup->addresses[i], aGroup->addressSizes[i]);
    }
}

/*******************************************************************************
 * countAddressesInAddressGroup takes an address group and returns the number
 * of addresses stored in that group.
 *
 * @param aGroup: address group that should be counted
 * @return Number of addresses in aGroup.
 ******************************************************************************/
u_int64_t countAddressesInAddressGroup(addressGroup *aGroup) {
    return aGroup->nAddresses;
}

/*******************************************************************************
 * initializeAddressGroups constructs a new struct of type addressGroups.
 * @return Pointer to the newly allocated and initialized struct.
 ******************************************************************************/
addressGroups *initializeAddressGroups() {
    addressGroups *aGroups = malloc(sizeof(addressGroups));
    aGroups->addressGroups = NULL;
    aGroups->nAddressGroups = 0;

    return aGroups;
}

/*******************************************************************************
 * freeAddressGroups takes a pointer to an address groups struct and destructs
 * that struct.
 *
 * @param aGroups: address groups struct that should be destructed
 * @param recursive: If set to 1, the address group items are destructed as well
 * @param freeSingleAddresses: If set to 1, the single addresses within each
 *        address group are freed.
 * @return NULL is returned.
 ******************************************************************************/
addressGroups *freeAddressGroups(addressGroups *aGroups, u_int64_t recursive, u_int64_t freeSingleAddresses) {
    if(recursive) {
        for(u_int64_t i = 0; i < aGroups->nAddressGroups; i++) {
            freeAddressGroup(aGroups->addressGroups[i], freeSingleAddresses);
            aGroups->addressGroups[i] = NULL;
        }
        aGroups->nAddressGroups = 0;
    }

    free(aGroups->addressGroups);
    free(aGroups);

    return NULL;
}

/*******************************************************************************
 * addAddressGroupToAddressGroups takes an address groups struct and an address
 * group and adds the address group the struct/list
 *
 * @param aGroups: Address groups the address group should be added to
 * @param aGroup: Group the should be added
 * @return modified address groups struct
 ******************************************************************************/
addressGroups *addAddressGroupToAddressGroups(addressGroups *aGroups, addressGroup *aGroup) {
    aGroups->nAddressGroups++;

    aGroups->addressGroups = realloc(aGroups->addressGroups, aGroups->nAddressGroups * sizeof(addressGroup));

    if(aGroup == NULL) {
        aGroup = initializeAddressGroup();
    }

    aGroups->addressGroups[aGroups->nAddressGroups - 1] = aGroup;

    return aGroups;
}

/*******************************************************************************
 * removeAddressGroupFromAddressGroups takes an address groups struct and an
 * index and removes the according address group.
 *
 * @param aGroups: address groups the address group should be removed from
 * @param index: Index of the group that should be removed
 * @param cleanupAddressGroups: If set to 1, the removed address group is
 *        destructed
 * @param cleanupAddress: If set to 1, the addresses in the address group are
 *        freed.
 * @return modified address groups struct
 ******************************************************************************/
addressGroups *removeAddressGroupFromAddressGroups(addressGroups *aGroups, u_int64_t index, u_int64_t cleanupAddressGroups, u_int64_t cleanupAddress) {
    aGroups->nAddressGroups--;

    if(cleanupAddressGroups) {
        freeAddressGroup(aGroups->addressGroups[index], cleanupAddress);
    }

    aGroups->addressGroups[index] = aGroups->addressGroups[aGroups->nAddressGroups];
    aGroups->addressGroups[aGroups->nAddressGroups] = NULL;

    return aGroups;
}

/*******************************************************************************
 * removeAddressGroupFromAddressGroupsWithoutRelocate takes an address groups
 * struct and removes the address group ad index. It does not relocate the array
 * but sets the corresponding pointer to NULL.
 *
 * @param aGroups: address groups the address group should be removed from
 * @param index: Index of the group that should be removed
 * @param cleanupAddressGroups: If set to 1, the removed address group is
 *        destructed
 * @param cleanupAddress: If set to 1, the addresses in the address group are
 *        freed.
 * @return modified address groups struct
 ******************************************************************************/
addressGroups *removeAddressGroupFromAddressGroupsWithoutRelocate(addressGroups *aGroups, u_int64_t index, u_int64_t cleanupAddressGroups, u_int64_t cleanupAddress) {
    if(cleanupAddressGroups) {
        freeAddressGroup(aGroups->addressGroups[index], cleanupAddress);
    }

    aGroups->addressGroups[index] = NULL;

    return aGroups;
}

/*******************************************************************************
 * cleanAddressGroups takes a address groups struct and relocates its address
 * group entries to remove the ones that are NULL
 *
 * @param aGroups: Address groups that should be cleaned up
 * @return modified address groups struct
 ******************************************************************************/
addressGroups *cleanAddressGroups(addressGroups *aGroups) {
    for(u_int64_t i = 0; i < aGroups->nAddressGroups; i++) {
        if(aGroups->addressGroups[i] == NULL || aGroups->addressGroups[i]->nAddresses == 0) {
            aGroups = removeAddressGroupFromAddressGroups(aGroups, i, 0, 0);
            i--;
        }
    }

    return aGroups;
}

/*******************************************************************************
 * printAddressGroups takes an address groups struct and dumps it to stdout.
 *
 * @param aGroups: Address groups that should be printed.
 ******************************************************************************/
void printAddressGroups(addressGroups *aGroups) {
    printf("Address Groups at %p contains %ld single address Groups:\n#######################\n", aGroups, aGroups->nAddressGroups);
    for(u_int64_t i = 0; i < aGroups->nAddressGroups; i++) {
        printf("%ld: ", i);
        printAddressGroup(aGroups->addressGroups[i]);
    }
}

/*******************************************************************************
 * countAddressesInAddressGroups takes an address groups struct and counts the
 * total number of addresses in that struct (e.g. the sum of all address groups)
 *
 * @param aGroups: Address groups struct that should be counted
 * @return Number of addresses
 ******************************************************************************/
u_int64_t countAddressesInAddressGroups(addressGroups *aGroups) {
    u_int64_t count = 0;
    for(u_int64_t i = 0; i < aGroups->nAddressGroups; i++) {
        count += countAddressesInAddressGroup(aGroups->addressGroups[i]);
    }

    return count;
}

/*******************************************************************************
 * parseNumber parses a decimal number string with the optional suffixes 'K',
 * 'M', or 'G' to a number.
 *
 * @param number: String that should be parsed
 * @return Number parsed from the string, 0 on error
 ******************************************************************************/
u_int64_t parseNumber(const char *number) {
    u_int64_t n = 0;
    u_int64_t i = 0;
    while(number[i] != 0x00) {
        n *= 10;

        if(number[i] >= '0' && number[i] <= '9') {
            n += number[i] - '0';
        }
        else if(number[i] == 'K') {
            return n * K;
        } else if(number[i] == 'M') {
            return n * M;
        } else if (number[i] == 'G') {
            return n * G;
        } else {
            return 0;
        }

        i++;
    }

    return n;
}

/*******************************************************************************
 * printMappingsAndTime prints mapping and timing information.
 *
 * @param aInfo: addrInfo struct that contains information about the mappings
 * @param timings: List of timings (one item for each item in aInfo)
 * @param len: Number of entries in aInfo and timings
 ******************************************************************************/
void printMappingsAndTime(addrInfo **aInfo, u_int64_t *timings, u_int64_t len) {
    u_int64_t digits = (u_int64_t)ceil(log10(len * 1.0));

    u_int64_t max = 0;
    for(u_int64_t i = 0; i < len; i++) {
        if(max < timings[i]) {
            max = timings[i];
        }
    }

    u_int64_t timeDigits = (u_int64_t)ceil(log10(max * 1.0));

    u_int64_t factor = max / 200;
    if(factor == 0) {
        factor = 1;
    }

    u_int64_t lastAddr = 0;

    u_int64_t colorSpecifier = 8;
    for(u_int64_t i = 0; i < len; i++) {
        if(aInfo[i]->pfn != lastAddr + 1) {
            // New group
            colorSpecifier++;
            if(colorSpecifier == 15) {
                colorSpecifier = 8;
            }
        }

        printf("\033[1m\033[38;5;%ldm[%0*ld]\033[0m PFN: 0x%016lx %0*ld: %.*s\n", colorSpecifier, (int)digits, i, aInfo[i]->pfn, (int)timeDigits, timings[i], (int)(timings[i]/factor), "##########################################################################################################################################################################################################");
        lastAddr = aInfo[i]->pfn;
    }
    printf("\033[0m");
}

/*******************************************************************************
 * printMappings takes a list of aInfo structs and prints the mapping
 * information.
 *
 * @param aInfo: List of addrInfo structs that contain the mapping information
 * @param len: Number of items in aInfo
 * @param debug: If set to 1, additional debug information are shown.
 ******************************************************************************/
void printMappings(addrInfo **aInfo, u_int64_t len, u_int64_t debug) {
    u_int64_t digits = (u_int64_t)ceil(log10(len * 1.0));
    u_int64_t lastAddr = 0;
    u_int64_t blockSize = 1;
    u_int64_t currentBlock = 1;
    u_int64_t cnt = 0;
    u_int64_t assumedBlockSize = 1;

    u_int64_t colorSpecifier = 8;
    for(u_int64_t i = 0; i < len; i++) {
        if(aInfo[i]->pfn != lastAddr + 1) {
            // New group
            colorSpecifier++;
            if(colorSpecifier == 15) {
                colorSpecifier = 8;
            }

            if(currentBlock > blockSize) {
                if(currentBlock == blockSize * 3) {
                    blockSize *= 2;
                } else {
                    blockSize = currentBlock;
                }
                printf("[%0*ld] Found new block size: %ld (x%ld)\n", (int)digits, i, blockSize, cnt);
                cnt = 1;
            } else if(currentBlock < blockSize){
                blockSize = currentBlock;
                printf("[%0*ld] Reduced block size: %ld (x%ld)\n", (int)digits, i, blockSize, cnt);
                cnt = 1;
            } else {
                cnt++;
                if(cnt >= 10 && assumedBlockSize != blockSize) {
                    assumedBlockSize = blockSize;
                    printf("\033[1m\033[38;5;40m[%0*ld] Assuming a block size of %ld\033[0m\n", (int)digits, i, assumedBlockSize);
                }
            }

            if(blockSize < assumedBlockSize) {
                printf("\033[1m\033[38;5;9m[%0*ld] Block size of %ld smaller than assumed block size of %ld\033[0m\n", (int)digits, i, blockSize, assumedBlockSize);
            }


            currentBlock = 1;
        } else {
            currentBlock++;
        }

        if(debug) {
            printf("\033[1m\033[38;5;%ldm[%0*ld]\033[0m HVA: 0x%016lx PFN: 0x%016lx\n", colorSpecifier, (int)digits, i, aInfo[i]->hva, aInfo[i]->pfn);
        }
        lastAddr = aInfo[i]->pfn;
    }
    printf("[%0*ld] Found blocks of size: %ld (x%ld)\n", (int)digits, len - 1, blockSize, cnt);
}

/*******************************************************************************
 * dumpPage takes a pointer to a page and dumps its content in hexadecimal.
 *
 * @param page: Pointer to the page that should be dumped.
 ******************************************************************************/
void dumpPage(void *page) {
    for(u_int64_t i = 0; i < sysconf(_SC_PAGESIZE); i++) {
        printf("%02x ", ((char*)page)[i]);
        if((i+1) % 8 == 0) {
            printf("  ");
        }
        if((i+1) % 64 == 0) {
            printf("\n");
        }
    }
    printf("\n");
}

/*******************************************************************************
 * updatePhysicalInformation takes a list of addrInfo structs and adds
 * physical addresses to them using /proc/self/pagemap
 *
 * @param aInfo: list of addrInfo structs that should get physical addresses
 *        added.
 * @param len: Number of items in aInfo
 * @return modified aInfo
 ******************************************************************************/
addrInfo **updatePhysicalInformation(addrInfo **aInfo, u_int64_t len) {
    if(addHpaToHva("/proc/self/pagemap", aInfo, len) != 0) {
        printf("Unable to get physical addresses.\n");
    }
    return aInfo;
}

/*******************************************************************************
 * addToArray takes an array and a value and adds the value to the end of that
 * array
 *
 * @param arr: Array the value should be added to
 * @param size: Pointer to the size of the array (is increased by 1 when a value
 *        is added).
 * @param val: Value that should be added
 * @return modified array
 ******************************************************************************/
u_int64_t *addToArray(u_int64_t *arr, u_int64_t *size, u_int64_t val) {
  (*size)++;
  arr = realloc(arr, sizeof(u_int64_t) * *size);
  arr[*size - 1] = val;
  return arr;
}

/*******************************************************************************
 * findValue searches a value within an array.
 *
 * @param arr: Array that should be searched
 * @param size: Number of elements in the array
 * @param val: Value that should be searched
 * @return On success, the index of the first occurrence of val is returned. If
 *         val was not found, -1 is returned.
 ******************************************************************************/
int64_t findValue(u_int64_t *arr, u_int64_t size, u_int64_t val) {
  for(u_int64_t i = 0; i < size; i++) {
    if(arr[i] == val) {
      return i;
    }
  }
  return -1;
}

/*******************************************************************************
 * uniqueAddToArray checks if the value that should be added is already in the
 * array. If that is the case, nothing is done. Otherwise, the value is added to
 * the array.
 *
 * @param arr: Array the value should be added to
 * @param size: Pointer to the size of the array (is increased when an element
 *        is added).
 * @param val: Value that should be added
 * @return Pointer to the array (that might be modified)
 ******************************************************************************/
u_int64_t *uniqueAddToArray(u_int64_t *arr, u_int64_t *size, u_int64_t val) {
  if(findValue(arr, *size, val) == -1) {
    return addToArray(arr, size, val);
  }
  return arr;
}
