#ifndef HLIB_UTIL_H
#define HLIB_UTIL_H

#define MODEL_AMD 0x01
#define MODEL_INTEL 0x02

#define HL_N_TIME_MEASUREMENTS 1000

#include<time.h>

#include<stdlib.h>
#include "../memlib/memoryInspect.h"
#include "asm.h"

typedef struct {
    addrInfo *aInfo;
    int64_t bank;
    int64_t row;
    int64_t page;
    int64_t tmpOffset;
} hlDimmItem;

typedef struct {
    long nItems;
    hlDimmItem **dItem;
} hlAddressGroup;

typedef struct  {
    long nItems;
    int blockSize;
    hlAddressGroup **aGroup;
} hlAddressGroups;

typedef struct {
    volatile char **aggressors;
    int *aggressorRows;
    volatile char **victims;
    int *victimRows;
    int nAggressors;
    int nVictims;
    int blockSize;
    int bank;
} hlHammerItem;

typedef struct {
    long nItems;
    hlHammerItem **hItem;
} hlHammerItems;

typedef struct {
    long nItems;
    long *masks;
    long *maskResults;
}hlMaskItems;

typedef struct {
    addrInfo **aInfo;
    long nItems;
}hlAddrInfos;

typedef struct {
    volatile char **aggressors;
    int nAggressors;
    volatile char *victim;
    int offset;
    int8_t flipMask;
    int flipMaskInverted;
} hlHammerLocation;

typedef struct {
    hlHammerLocation **hLocation;
    long nItems;
} hlHammerLocations;


int hlHammerMemcmp (const void * s1 , const void * s2 , size_t len);
hlMaskItems *hlConstructMaskItems();
hlMaskItems *hlDestructMaskItems(hlMaskItems *mItems);
int hlAddMaskToMaskItems(hlMaskItems *mItems, long mask, long maskResult, pthread_mutex_t *mutex);
hlHammerItem *hlConstructHammerItem(hlDimmItem **aggressors, hlDimmItem **victims, int nAggressors, int nVictims, int blockSize);
hlHammerItems *hlConstructHammerItems();
hlHammerItems *hlMergeHammerItems(hlHammerItems *h1, hlHammerItems *h2, int verbosity);
hlHammerItem *hlDestructHammerItem(hlHammerItem *hItem);
hlHammerItems *hlDestructHammerItems(hlHammerItems *hItems);
hlHammerItems *hlAddHammerItem(hlHammerItems *hItems, hlDimmItem **aggressors, hlDimmItem **victims, int nAggressors, int nVictims, int blockSize);
hlDimmItem *hlConstructDimmItem(addrInfo *aInfo);
hlDimmItem *hlDestructDimmItem(hlDimmItem *dItem);
hlAddressGroup *hlConstructAddressGroup();
hlAddressGroup *hlDestructAddressGroup(hlAddressGroup *aGroup);
hlAddressGroups *hlConstructAddressGroups(int nBanks, int blockSize);
hlAddressGroups *hlDestructAddressGroups(hlAddressGroups *aGroups);
int hlAddAddressToAddressGroup(hlAddressGroup *aGroup, addrInfo *aInfo);
int hlMeasureAccessTimeRdtscp(volatile char *ptr1, volatile char *ptr2, int iter, int fenced);
int hlMeasureAccessTimeGettime(volatile char *ptr1, volatile char *ptr2, int iter, int fenced);
int hlCompareAGroupBySizeReverse(const void *a1, const void *a2);
int hlCompareDimmItemByHva(const void *a1, const void *a2);
int hlCompareDimmItemByPfn(const void *a1, const void *a2);
int hlCompareDimmItemByGva(const void *a1, const void *a2);
int hlCompareDimmItemByGfn(const void *a1, const void *a2);
int hlCompareDimmItemByRow(const void *a1, const void *a2);
int hlCompareDimmItemByBank(const void *a1, const void *a2);
int hlCompareInt(const void *a1, const void *a2);
int hlCompareLong(const void *a1, const void *a2);
void hlPrintAddressGroupStats(hlAddressGroups *aGroups, int verbose);
int hlExportAddressGroupStats(hlAddressGroups *aGroups, const char *filename);
int hlExportAccessTimeDistribution(int *distribution, int size, const char *filename, int verbosity);
long hlGetNextBitmask(long x);
void hlPrintBinary(unsigned long v, int verbosity);
void hlPrintBinaryPfn(unsigned long pfn, int bankBits, int verbosity);
int64_t hlGetBankFromPfn(int64_t pfn, hlMaskItems *mItems);
int64_t hlGetRowFromPfn(int64_t pfn, hlMaskItems *mItems);
int64_t hlGetPageFromPfn(int64_t pfn);
void hlPrintTimingForRow(int measurements, volatile char *ptr, int scale, int getTime, int fenced, int verbosity);
int hlImportConfig(char *filename, int *nBanks, hlMaskItems **mItems, int verbosity);
int hlExportConfig(char *filename, int nBanks, hlMaskItems *mItems, int verbosity);
hlHammerLocation *hlConstructHammerLocation(volatile char **aggressors, int nAggressors, volatile char *victim, int offset, int translateToPhysical, int verbosity, int8_t flipMask, int flipMaskInverted);
hlHammerLocation *hlDestructHammerLocation(hlHammerLocation *hLocation);
hlHammerLocations *hlConstructHammerLocations();
hlHammerLocations *hlDestructHammerLocations(hlHammerLocations *hLocations);
int hlAddItemToHammerLocations(hlHammerLocations *hLocations, volatile char **aggressors, int nAggressors, volatile char *victim, int offset, int translateToPhysical, pthread_mutex_t *mutex, int verbosity, int8_t flipMask, int flipMaskInverted);
int hlAreHammerLocationsEqual(hlHammerLocation *i1, hlHammerLocation *i2);
int hlExportHammerLocations(char *filename, hlHammerLocations *hLocations, int verbosity);
int hlImportHammerLocations(char *filename, hlHammerLocations *hLocations, int verbosity);
char *hlGetRandomPages(int nPages);
char *hlGetRandomPage();


/**
 * hlXorBits taks a value and calculates the XOR value of all bits in
 * this value (if there is an even or odd number of bits)
 *
 * @param x value the XOR-value should be calculated for
 * @return XOR-value of x
 */
static inline int hlXorBits(long x) {
    int sum = 0;
    while(x != 0) {
        //x&-x has a binary one where the lest significant one
        //in x was before. By applying XOR to this, the last
        //one becomes a zero.
        //
        //So, this overwrites all ones in x and toggles sum
        //every time until there are no ones left.
        //
        //This looks a bit strange but increases speed. Because
        //this is called very often (once for each mask and each
        //pfn), it should be done this way. Maybe, there is also
        //an even better way.
        sum^=1;
        x ^= (x&-x);
    }
    return sum;
}

/**
 * hlCountBits takes a value and counts the number of set bits in this
 * value.
 *
 * @param x value the sum should be calculated for
 * @return sum of set bits in x
 */
static inline int hlCountBits(long x) {
    int sum = 0;
    while(x != 0) {
        sum++;
        x ^= (x&-x);
    }
    return sum;
}

static inline int hlGetTimingRdtscp(volatile char *ptr, int fenced) {
    void(*mfence)() = hlDummy_mfence;
    void(*lfence)() = hlDummy_lfence;
    void(*cpuid)() = hlDummy_cpuid;
    if(fenced) {
        mfence = hlReal_mfence;
        lfence = hlReal_lfence;
        cpuid = hlReal_cpuid;
    }
    long t1, t2;
    hlClflush(ptr);
    mfence();
    cpuid();
    t1 = hlRdtscp();
    *ptr;
    lfence();
    cpuid();
    t2 = hlRdtscp();
    mfence();
    return t2-t1;
}

static inline int hlGetTimingGettime(volatile char *ptr, int fenced) {
    void(*mfence)() = hlDummy_mfence;
    void(*lfence)() = hlDummy_lfence;
    void(*cpuid)() = hlDummy_cpuid;
    if(fenced) {
        mfence = hlReal_mfence;
        lfence = hlReal_lfence;
        cpuid = hlReal_cpuid;
    }
    struct timespec t1, t2;
    hlClflush(ptr);
    mfence();
    cpuid();
    clock_gettime(CLOCK_MONOTONIC, &t1);
    *ptr;
    lfence();
    cpuid();
    clock_gettime(CLOCK_MONOTONIC, &t2);
    mfence();
    return (t2.tv_sec - t1.tv_sec) * 1000000000 + (t2.tv_nsec - t1.tv_nsec);;
}

static inline long hlLongFromHex(char *hex) {
    long v = 0;
    for(int i = 0; hex[i]!=0; i++) {
        v *= 16;
        if(hex[i]>='0' && hex[i] <='9') {
            v += hex[i] - '0';
        } else if(hex[i]>='a' && hex[i] <= 'f') {
            v += hex[i] - 'a' + 10;
        }
    }
    return v;
}
#endif
