#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<malloc.h>
#include<sys/mman.h>
#include<string.h>
#include<fcntl.h>
#include<errno.h>

#include "group.h"
#include "bankSequence.h"
#include "helper.h"
#include "memlib/util.h"

/*******************************************************************************
 * calculateBankFromAddress takes an address, a list of address functions and
 * the number of functions and applies the functions to the submitted address.
 * It returns the number of the bank calculated based on the functions.
 *
 * @param address: Address the bank number should be calculated for
 * @param fns: List of address functions
 * @param nFns: Number of address functions in the list
 * @return Number of the bank calculated for the submitted address based on the
 *         address functions.
 ******************************************************************************/
u_int64_t calculateBankFromAddress(volatile char *address, u_int64_t *fns, u_int64_t nFns) {
  u_int64_t bank = 0;
  for(u_int64_t i = 0; i < nFns; i++) {
    bank *= 2;
    bank += xorBits((u_int64_t)address & fns[i]);
  }

  return bank;
}

/*******************************************************************************
 * getBankSequence calculates the bank sequence based on the submitted address
 * functions.
 *
 * @param len: Pointer to store the length of the sequence calculated
 * @param fns: List of address functions
 * @param nFns: Number of address functions in the list
 * @return Sequence of bank numbers
 ******************************************************************************/
u_int64_t *getBankSequence(u_int64_t *len, u_int64_t *fns, u_int64_t nFns) {
  u_int64_t maxFnsBit = 0;
  u_int64_t fnsMask = 0x00;

  for(u_int64_t i = 0; i < nFns; i++) {
    fnsMask |= fns[i];
  }

  for(u_int64_t i = 0; i < 64; i++) {
    if(fnsMask & (1UL<<i)) {
      maxFnsBit = i;
    }
  }

  *len = (1UL<<(maxFnsBit+1-12)) * 2;
  u_int64_t *bankSequence = malloc(sizeof(u_int64_t) * *len);

  u_int64_t i = 0;
  for(u_int64_t address = 0x00; address < *len * sysconf(_SC_PAGESIZE); address += sysconf(_SC_PAGESIZE)) {
    bankSequence[i++] = calculateBankFromAddress((volatile char *)address, fns, nFns);
  }

  return bankSequence;
}

uniqueBankOffset *getBankOffsetForBlockSize(bankInformation *bInfo, u_int64_t blockSize) {
  uniqueBankOffset *uBankOffset = NULL;

  // Search for an offset item that matches the requested blocksize
  for(u_int64_t offsetIdx = 0; offsetIdx < bInfo->nUniqueBankOffsets; offsetIdx++) {
    if(bInfo->uniqueBankOffsets[offsetIdx]->sequenceLength == blockSize) {
      uBankOffset = bInfo->uniqueBankOffsets[offsetIdx];
    }
  }

  // Check if the requested block size is a multiple of the maximum block
  // size
  if(blockSize > bInfo->uniqueBankOffsets[bInfo->nUniqueBankOffsets - 1]->sequenceLength && blockSize % bInfo->uniqueBankOffsets[bInfo->nUniqueBankOffsets - 1]->sequenceLength == 0) {
    uniqueBankOffset *tmpBankOffset = bInfo->uniqueBankOffsets[bInfo->nUniqueBankOffsets - 1];

    uBankOffset = malloc(sizeof(uniqueBankOffset));
    uBankOffset->nOffsets = tmpBankOffset->nOffsets;
    uBankOffset->sequenceLength = tmpBankOffset->sequenceLength;
    uBankOffset->verificationOffset = tmpBankOffset->verificationOffset + blockSize - tmpBankOffset->sequenceLength;
    uBankOffset->offsets = malloc(sizeof(u_int64_t) * tmpBankOffset->nOffsets);
    for(u_int64_t i = 0; i < tmpBankOffset->nOffsets; i++) {
      uBankOffset->offsets[i] = tmpBankOffset->offsets[i];
    }
  }

  if(uBankOffset == NULL) {
    printf("\nUnable to find offsets for block size %ld. This should not happen.\n", blockSize);
  }
  return uBankOffset;
}

/*******************************************************************************
 * verifyBankSequence takes bankInformation, a list of banks, an additional
 * verification bank, the current block size, a part sequence and flags. It
 * calculates if the submitted banks and verificationBank are valid
 *
 * @param bInfo: Bank information calculated with getBankInformation
 * @param banks: List of banks that have to match
 * @param nBanks: Number of banks in the list of banks
 * @param verificationBank: Bank used for verification
 * @param blockSize: Current block size (which implicitely specifies the offsets
 *        of the banks and verificationBank)
 * @param partSequence: Sequence of banks that matched (usually used to group
 *        the addresses afterwards)
 * @param flags: Additional flags
 * @return On success, 1 is returned. Otherwise, 0 is returned.
 ******************************************************************************/
u_int64_t verifyBankSequence(bankInformation *bInfo, u_int64_t *banks, u_int64_t nBanks, u_int64_t verificationBank, u_int64_t blockSize, u_int64_t **partSequence, flag_t flags) {
  u_int64_t *sequence = NULL;
  if(partSequence == NULL) {
    sequence = malloc(sizeof(u_int64_t) * blockSize);
    partSequence = &sequence;
  }

  u_int64_t nMatches = 0;
  uniqueBankOffset *uBankOffset = getBankOffsetForBlockSize(bInfo, blockSize);

  if(uBankOffset == NULL) {
    return 0;
  }


  if(uBankOffset->nOffsets != nBanks) {
    printf("The number of banks submitted (%ld) does not match the number of offsets required (%ld).\n", nBanks, uBankOffset->nOffsets);
    return 0;
  }

  for(u_int64_t i = 0; i < bInfo->nBankSequence; i++) {
    u_int64_t allOffsetsMatch = 1;
    for(u_int64_t offsetIdx = 0; offsetIdx < uBankOffset->nOffsets; offsetIdx++) {
      if(bInfo->bankSequence[(i + uBankOffset->offsets[offsetIdx]) % bInfo->nBankSequence] != banks[offsetIdx]) {
        allOffsetsMatch = 0;
      }
    }

    if(allOffsetsMatch) {
      if(nMatches == 0) {
        for(u_int64_t j = 0; j < blockSize; j++) {
          (*partSequence)[j] = bInfo->bankSequence[(i+j) % bInfo->nBankSequence];
        }
        nMatches++;
      } else {
        u_int64_t matchIsNew = 1;
        for(u_int64_t j = 0; j < blockSize; j++) {
          if((*partSequence)[j] != bInfo->bankSequence[(i+j) % bInfo->nBankSequence]) {
            matchIsNew = 0;
          }
        }
        if(!matchIsNew) {
          if(flags & FL_DEBUG) {
            printf("[DEBUG]: New match: ");
            for(u_int64_t j = 0; j < blockSize; j++) {
              printf("%ld ", bInfo->bankSequence[(i+j) % bInfo->nBankSequence]);
            }
            printf("\n");
          }
          nMatches++;
        }
      }
    }
  }

  u_int64_t retVal = 1;

  if(nMatches != 1) {
    if(flags & FL_DEBUG) {
      printf("There should be exactly one match for the submitted sequence of length %ld. Found %ld:", blockSize, nMatches);
      for(u_int64_t i = 0; i < uBankOffset->nOffsets; i++) {
        printf(" %ld", uBankOffset->offsets[i]);
      }
      printf(" (v: %ld), searched for", uBankOffset->verificationOffset);
      for(u_int64_t i = 0; i < uBankOffset->nOffsets; i++) {
        printf(" %ld", banks[i]);
      }
      printf(" (v: %ld)\n", verificationBank);

    }
    retVal = 0;
  }

  if((*partSequence)[uBankOffset->verificationOffset] != verificationBank && retVal == 1) {
    if(flags & FL_DEBUG) {
      printf("The verification bank at offset %ld does not match %ld != %ld\n", uBankOffset->verificationOffset, (*partSequence)[uBankOffset->verificationOffset], verificationBank);
      printf("part sequence:");
      for(u_int64_t i = 0; i < blockSize; i++) {
        printf(" %ld", (*partSequence)[i]);
      }
      printf("\n");
    }
    retVal = 0;
  }

  if(sequence != NULL) {
    free(sequence);
  }

  return retVal;
}

/*******************************************************************************
 * getBankMapping calculates the mapping between group indices and bank numbers.
 *
 * @param aGroups: Address groups that should be mapped (e.g. indices of these
 *        groups are used).
 * @param fns: List of address functions.
 * @param nFns: Number of address functions in the list.
 * @param threshold: Threshold value between row hit and row conflict.
 * @param maxGroupComparisons: Maximum number of groups comparisons (e.g. how
 *        many address are compared from each group, if there are at least that
 *        many addresses in the group)
 * @param nHugePages: Number of hube page that should be allocated. Huge pages
 *        are required to get a stable mapping (e.g. 512 continuous pages)
 * @param measurementsPerAddress: Number of measurements for each address
 * @return Mapping of group indices to bank numbers.
 ******************************************************************************/
u_int64_t *getBankMapping(addressGroups *aGroups, u_int64_t *fns, u_int64_t nFns, u_int64_t threshold, u_int64_t maxGroupComparisons, u_int64_t nHugePages, u_int64_t measurementsPerAddress) {
  u_int64_t nGroups = aGroups->nAddressGroups;
  u_int64_t *mapping = malloc(sizeof(u_int64_t) * nGroups * nGroups);

  u_int64_t invalidMappings = 1;
  invalidMappings = 0;

  for(u_int64_t i = 0; i < nGroups * nGroups; i++) {
    mapping[i] = 0;
  }

  addressGroup *hugePage = getHugepages(nHugePages);

  for(u_int64_t i = 0; i < nHugePages * N_PAGES_FOR_THP; i++) {
    volatile char *currentPage = hugePage->addresses[i / 512] + (i % 512) * sysconf(_SC_PAGESIZE);
    int64_t group = -1;

    int mustGroup = 0;
    while(group == -1) {
      group = getGroupIdxForAddress(aGroups, maxGroupComparisons, threshold, currentPage, measurementsPerAddress, FL_NONE, mustGroup);
    }

    // Address currentPage belongs to the group at Idx. Now we can calculate
    // the "real" group index based on the address and store it as possible
    // mapping

    u_int64_t bank = calculateBankFromAddress(currentPage, fns, nFns);

    mapping[group * nGroups + bank] += 1;
  }

  u_int64_t *finalMapping = malloc(sizeof(u_int64_t) * nGroups);
  for(u_int64_t group = 0; group < nGroups; group++) {
    u_int64_t maxBank = 0;
    u_int64_t value = 0;
    for(u_int64_t bank = 0; bank < nGroups; bank++) {
      if(mapping[group * nGroups + bank] > value) {
        if(value > 0) {
          invalidMappings += value;
          printf("\033[1m\033[38;5;9mInvalidating mapping %ld <-> %ld (x%ld)\033[0m\n", group, bank, value);
        }
        printf("\033[1m\033[38;5;40mSetting mapping %ld <-> %ld (x%ld)\033[0m\n", group, bank, mapping[group * nGroups + bank]);
        maxBank = bank;
        value = mapping[group * nGroups + bank];
      } else if(mapping[group * nGroups + bank] > 0) {
        invalidMappings += mapping[group * nGroups + bank];
        printf("\033[1m\033[38;5;9mInvalidating mapping %ld <-> %ld (x%ld)\033[0m\n", group, bank, mapping[group * nGroups + bank]);
      }
    }

    finalMapping[group] = maxBank;
  }

  freeAddressGroup(hugePage, 1);
  free(mapping);

  printf("%ld mappings did not match.\n", invalidMappings);

  // Beware: magical number. It worked but not sure if it is the best
  // possibility.
  if(invalidMappings >= 64) {
    return NULL;
  }

  return finalMapping;
}

/*******************************************************************************
 * getReverseBankMapping takes a bank mapping and reverses it, e.g. takes a
 * group index -> bank number mapping and calculates a bank number -> group
 * index mapping from it.
 *
 * @param bankMapping: group index -> bank number mapping calculate before
 * @param len: Number of entries in the mapping
 * @return Reversed mapping
 ******************************************************************************/
u_int64_t *getReverseBankMapping(u_int64_t *bankMapping, u_int64_t len) {
  printf("In getReverseBankMapping. len=%ld\n", len);
  u_int64_t *reverseMapping = malloc(sizeof(u_int64_t) * len);
  for(u_int64_t i = 0; i < len; i++) {
    reverseMapping[bankMapping[i]] = i;
  }

  return reverseMapping;
}

u_int64_t *getNextOffsets(u_int64_t *offset, u_int64_t *len, u_int64_t nSequence) {
  u_int64_t lastMatchForLen = 1;
  for(u_int64_t idx = 0; idx < *len; idx++) {
    if(offset[idx] < nSequence - *len) {
      lastMatchForLen = 0;
    }
  }

  // Increase the number of offsets and inizialize them as the *len lowest
  // numbers
  if(lastMatchForLen) {
    (*len)++;
    offset = realloc(offset, *len * sizeof(u_int64_t));
    for(u_int64_t idx = 0; idx < *len; idx++) {
      offset[idx] = idx + 1;
    }

    return offset;
  }

  for(u_int64_t idx = 1; idx <= *len; idx++) {
    if(offset[idx] != offset[idx - 1] + 1 || idx == *len) {
      offset[idx-1]++;
      for(u_int64_t iidx = 0; iidx < idx - 1; iidx++) {
        offset[iidx] = iidx + 1;
      }
      return offset;
    }
  }

  printf("This should never happen!\n");
  free(offset);
  return NULL;
}

/*******************************************************************************
 * getUniqueSubSequences takes a sequence and calculates offsets where unique
 * sub sequences start
 *
 * @param sequence: Sequence that should be inspected
 * @param nSequence: Length of the sequence
 * @param requiredOffset: Size of the searched sub sequences
 * @param nMatches: Pointer where the number of results is stored
 * @return List of offsets where unique sub sequences start
 ******************************************************************************/
u_int64_t *getUniqueSubSequences(u_int64_t *sequence, u_int64_t nSequence, u_int64_t requiredOffset, u_int64_t *nMatches) {
  u_int64_t *matches = NULL;
  *nMatches = 0;

  for(u_int64_t startIdx = 0; startIdx < nSequence; startIdx++) {
    u_int64_t subSequenceIsNew = 1;

    for(u_int64_t offsetIdx = 0; offsetIdx < *nMatches && subSequenceIsNew; offsetIdx++) {
      u_int64_t digitsMatch = 1;
      for(u_int64_t idx = 0; idx < requiredOffset && idx < nSequence; idx++) {
        if(sequence[(startIdx + idx) % nSequence] != sequence[(matches[offsetIdx] + idx) % nSequence]) {
          digitsMatch = 0;
        }
      }

      if(digitsMatch) {
        subSequenceIsNew = 0;
      }
    }

    if(subSequenceIsNew) {
      (*nMatches)++;
      matches = realloc(matches, *nMatches * sizeof(u_int64_t));
      matches[*nMatches-1] = startIdx;
    }
  }

  return matches;
}

/*******************************************************************************
 * getOffsetsForUniqueDetection calculates offsets that are required for a given
 * sequence and sub sequence length to find only unique sub sequences, e.g.
 * either all sub sequences are the same at the offsets and anywhere else or
 * they are different at these offsets as well, so measuring at the offsets is
 * sufficient to uniquely identify a sub sequence of given lenght.
 *
 * @param sequence: Sequence that should be anlyzed
 * @param nSequence: Length of the sequence
 * @param requiredOffset: Length of the sub sequences that ar searched
 * @param debug: Print additional debug information
 * @param nOffsets: Pointer to store the number of results
 * @return List of offsets that is required to uniquely identify a sub sequence
 *         of length requiredOffset within sequence.
 ******************************************************************************/
u_int64_t *getOffsetsForUniqueDetection(u_int64_t *sequence, u_int64_t nSequence, u_int64_t requiredOffset, u_int64_t debug, u_int64_t *nOffsets) {
  u_int64_t nMatches = 0;
  u_int64_t *uniqueMatches = getUniqueSubSequences(sequence, nSequence, requiredOffset, &nMatches);

  // Find matches where the first and requiredOffset_th element are equal, so
  // they would both fullfill the requiremement to match but are not entirely
  // equal (entirely equal matches were already filtered by
  // findUniqueSequences).
  *nOffsets = 0;
  u_int64_t *offsets = NULL;
  while(offsets != NULL || *nOffsets == 0) {
    u_int64_t offsetsValid = 1;
    for(u_int64_t firstMatchIdx = 0; firstMatchIdx < nMatches; firstMatchIdx++) {
      for(u_int64_t secondMatchIdx = firstMatchIdx + 1; secondMatchIdx < nMatches; secondMatchIdx++) {
        u_int64_t sequencesMatchOffsets = sequence[(uniqueMatches[firstMatchIdx]) % nSequence] == sequence[(uniqueMatches[secondMatchIdx]) % nSequence];
        for(u_int64_t offsetIdx = 0; offsetIdx < *nOffsets && sequencesMatchOffsets == 1; offsetIdx++) {
          u_int64_t offset = offsets[offsetIdx];
          sequencesMatchOffsets = sequencesMatchOffsets && sequence[(uniqueMatches[firstMatchIdx] + offset) % nSequence] == sequence[(uniqueMatches[secondMatchIdx] + offset) % nSequence];
        }

        if(sequencesMatchOffsets) {
          u_int64_t sequencesAreDifferent = 0;
          for(u_int j = 0; j < requiredOffset && j < nSequence; j++) {
            if(sequence[(uniqueMatches[firstMatchIdx] + j) % nSequence] != sequence[(uniqueMatches[secondMatchIdx] + j) % nSequence]) {
              sequencesAreDifferent = 1;
              break;
            }
          }

          if(sequencesAreDifferent) {
            offsetsValid = 0;
            if(debug) {
              printf("No match for sequences (%ld):", requiredOffset);
              for(u_int64_t i = 0; i < requiredOffset && i < nSequence; i++) {
                printf(" %02ld", sequence[(uniqueMatches[firstMatchIdx] + i) % nSequence]);
              }

              printf(" (");

              for(u_int64_t i = 0; i < requiredOffset && i < nSequence; i++) {
                printf(" %02ld", sequence[(uniqueMatches[secondMatchIdx] + i) % nSequence]);
              }

              printf(")\n");
            }
          } else {
            printf("[WARNING]: Sequences are equal at offsets %ld, %ld for block size %ld, this should not happen.\n", firstMatchIdx, secondMatchIdx, requiredOffset);
            exit(0);
          }
        }
      }
    }

    if(offsetsValid) {
      // Include 0 as offset
      offsets = realloc(offsets, sizeof(u_int64_t) * (*nOffsets + 1));
      for(u_int64_t offsetIdx = *nOffsets; offsetIdx > 0; offsetIdx--) {
        offsets[offsetIdx] = offsets[offsetIdx-1];
      }
      offsets[0] = 0;
      *nOffsets += 1;
      return offsets;
    }
    offsets = getNextOffsets(offsets, nOffsets, requiredOffset>nSequence?nSequence:requiredOffset);
  }

  return NULL;
}

/*******************************************************************************
 * getOffsetForVerification calculates an additional verification offset that
 * can be used to verify that the block size is still the same.
 *
 * @param sequence: Sequence that should be analyzed
 * @param nSequence: Length of the sequence
 * @param offsets: Offsets that can be used to uniquely identify a sub sequence
 *        (see getOffsetsForUniqueDetection())
 * @param nOffsets: Number of offsets submitted
 * @param nBanks: Number of banks in the system
 * @param requiredOffset: Length of the sub sequences that should be uniquely
 *        verified.
 * @return Index that can be used for additional unique verification of sub
 *         sequences
 ******************************************************************************/
u_int64_t getOffsetForVerification(u_int64_t *sequence, u_int64_t nSequence, u_int64_t *offsets, u_int64_t nOffsets, u_int64_t nBanks, u_int64_t requiredOffset) {
  u_int64_t distribution[nBanks];
  u_int64_t bestValue = -1;
  u_int64_t bestOffset = -1;

  u_int64_t nMatches = 0;
  u_int64_t *uniqueMatches = getUniqueSubSequences(sequence, nSequence, requiredOffset, &nMatches);

  // Search in the 2nd half of the block for the verification offset. This is
  // useful when multiple block sizes have the same offsets in order to 
  // distinguish e.g. between 32 and 64 (otherwise, they would have the same
  // parameters so there would be no way to distinguish between them).
  for(u_int64_t currentOffset = requiredOffset/2; currentOffset < requiredOffset; currentOffset++) {
    u_int64_t offsetValid = 1;
    for(u_int64_t i = 0; i < nOffsets; i++) {
      if(offsets[i] == currentOffset) {
        offsetValid = 0;
        break;
      }
    }

    if(offsetValid == 0) {
      continue;
    }

    for(u_int64_t i = 0; i < nBanks; i++) {
      distribution[i] = 0;
    }

    for(u_int64_t matchIdx = 0; matchIdx < nSequence; matchIdx++) {
      distribution[sequence[(uniqueMatches[matchIdx] + currentOffset) % nSequence]]++;
    }

    u_int64_t sum = 0;
    for(u_int64_t i = 0; i < nBanks; i++) {
      sum += distribution[i];
    }

    u_int64_t mean = sum / nBanks;

    sum = 0;

    for(u_int64_t i = 0; i < nBanks; i++) {
      sum += (distribution[i] - mean) * (distribution[i] - mean);
    }

    u_int64_t variance = sum / nBanks;

    if(variance < bestValue || bestValue == -1) {
      bestValue = variance;
      bestOffset = currentOffset;
    }
  }

  return bestOffset;
}

/*******************************************************************************
 * dumpUniqueBankOffsets takes a list of unique bank offsets and prints them
 * to stdout.
 *
 * @param bOffset Unique bank offsets that should be printed.
 ******************************************************************************/
void dumpUniqueBankOffsets(uniqueBankOffset *bOffset) {
  printf("Unique bank offset. verificationOffset=%ld sequenceLength=%ld offsets=", bOffset->verificationOffset, bOffset->sequenceLength);
  for(u_int64_t i = 0; i < bOffset->nOffsets; i++) {
    printf("%ld ", bOffset->offsets[i]);
  }
  printf("\n");
}

/*******************************************************************************
 * handleLine parses a line from a temporary cache file and stores the
 * parsed information in the submitted bank information struct.
 *
 * @param line: Line that should be parsed
 * @param bInfo: bInfo struct where the parsed information should be stored
 * @return Modified bank information struct
 ******************************************************************************/
bankInformation *handleLine(char *line, bankInformation * bInfo) {
  char *key = strtok(line, "=");
  char *value = strtok(NULL, "=");

  if(key == NULL || value == NULL) {
    return bInfo;
  }

  if(strcmp(key, "bankSequence") == 0) {
    bInfo->bankSequence = NULL;
    bInfo->nBankSequence = 0;
    char *sItem = strtok(value, ",");
    while(sItem != NULL) {
      bInfo->nBankSequence ++;
      bInfo->bankSequence = realloc(bInfo->bankSequence, sizeof(u_int64_t) * bInfo->nBankSequence);
      bInfo->bankSequence[bInfo->nBankSequence - 1] = atoi(sItem);
      sItem = strtok(NULL, ",");
    }
  }

  if(strcmp(key, "uniqueBankOffset") == 0) {
    bInfo->nUniqueBankOffsets ++;
    bInfo->uniqueBankOffsets = realloc(bInfo->uniqueBankOffsets, sizeof(uniqueBankOffset) * bInfo->nUniqueBankOffsets);

    uniqueBankOffset *uBankOffset = malloc(sizeof(uniqueBankOffset));
    uBankOffset->offsets = NULL;
    uBankOffset->nOffsets = 0;
    uBankOffset->verificationOffset = 0;
    uBankOffset->sequenceLength = 0;

    char *verificationOffset = strtok(value, ";");
    char *sequenceLength = strtok(NULL, ";");
    char *offsets = strtok(NULL, ";");

    char *sItem = strtok(offsets, ",");
    while(sItem != NULL) {
      uBankOffset->nOffsets ++;
      uBankOffset->offsets = realloc(uBankOffset->offsets, sizeof(u_int64_t) * uBankOffset->nOffsets);
      uBankOffset->offsets[uBankOffset->nOffsets - 1] = atoi(sItem);
      sItem = strtok(NULL, ",");
    }

    uBankOffset->verificationOffset = atoi(verificationOffset);
    uBankOffset->sequenceLength = atoi(sequenceLength);

    bInfo->uniqueBankOffsets[bInfo->nUniqueBankOffsets - 1] = uBankOffset;
  }

  return bInfo;
}

/*******************************************************************************
 * loadBankInformationFromCache takes a file path and tries to load the bank
 * information from the specified file.
 *
 * @param filepath: Path to the file that contains the cached data.
 * @return bank information struct that contains the loaded information
 ******************************************************************************/
bankInformation *loadBankInformationFromCache(char *filepath) {
  char *content = readFile(filepath);
  if(content == NULL) {
    return NULL;
  }

  bankInformation *bInfo = malloc(sizeof(bankInformation));
  bInfo->bankSequence = NULL;
  bInfo->nBankSequence = 0;
  bInfo->uniqueBankOffsets = NULL;
  bInfo->nUniqueBankOffsets = 0;
  bInfo->bankToGroupMapping = NULL;
  bInfo->groupToBankMapping = NULL;
  bInfo->nBanks = 0;

  char *line = content;
  u_int64_t offset = -1;

  while((line = strtok(line + offset + 1, "\n")) != NULL) {
    offset = strlen(line);
    bInfo = handleLine(line, bInfo);
  }

  free(content);

  return bInfo;
}

/*******************************************************************************
 * storeBankInformationToCache takes a path and a bankInformation struct and
 * dumps the information from the struct to the specified file.
 *
 * @param filepath: Path of the file that should contain the cached information.
 * @param bInfo: bInfo struct that should be cached.
 * @return On success, 1 is returned. Otherwise, 0 is returned.
 ******************************************************************************/
u_int64_t storeBankInformationToCache(char *filepath, bankInformation *bInfo) {
  int fd = open(filepath, O_WRONLY|O_TRUNC|O_CREAT, 0640);
  if(fd == -1) {
    printf("Unable to open file %s. Error: %s\n", filepath, strerror(errno));
    return 0;
  }

  if(dprintf(fd, "bankSequence=%ld", bInfo->bankSequence[0]) < 0) {
    printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
    return 0;
  }
  for(u_int64_t i = 1; i < bInfo->nBankSequence; i++) {
    if(dprintf(fd, ",%ld", bInfo->bankSequence[i]) < 0) {
      printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
      return 0;
    }
  }

  for(u_int64_t j = 0; j < bInfo->nUniqueBankOffsets; j++) {
    if(dprintf(fd, "\nuniqueBankOffset=") < 0) {
      printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
      return 0;
    }
    if(dprintf(fd, "%ld", bInfo->uniqueBankOffsets[j]->verificationOffset) < 0) {
      printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
      return 0;
    }

    if(dprintf(fd, ";%ld", bInfo->uniqueBankOffsets[j]->sequenceLength) < 0) {
      printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
      return 0;
    }

    if(bInfo->uniqueBankOffsets[j]->nOffsets > 0) {
      if(dprintf(fd, ";%ld", bInfo->uniqueBankOffsets[j]->offsets[0]) < 0) {
        printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
        return 0;
      }
    }
    for(u_int64_t i = 1; i < bInfo->uniqueBankOffsets[j]->nOffsets; i++) {
      if(dprintf(fd, ",%ld", bInfo->uniqueBankOffsets[j]->offsets[i]) < 0) {
        printf("Unable to write to file %s. Error: %s\n", filepath, strerror(errno));
        return 0;
      }
    }

  }

  return 1;
}

/*******************************************************************************
 * getBankInformation takes address functions and flags and tries to load the
 * bank information from a cache file "bankInformation.txt". If that is not
 * possible, it measures the bank information.
 *
 * @param fns: Address functions
 * @param nFns: Number of address functions
 * @param flags: Additional flags
 * @return bankInformation struct that can be used further.
 ******************************************************************************/
bankInformation *getBankInformation(u_int64_t *fns, u_int64_t nFns, flag_t flags) {
  if((flags & FL_NOCACHE) == 0) {
    bankInformation *bInfo = loadBankInformationFromCache("bankInformation.txt");

    if(bInfo != NULL) {
      u_int64_t nBankSequence = 0;
      u_int64_t *bankSequence = getBankSequence(&nBankSequence, fns, nFns);

      if(nBankSequence == bInfo->nBankSequence) {
        u_int64_t allValid = 1;
        for(u_int64_t i = 0; i < nBankSequence; i++) {
          if(bankSequence[i] != bInfo->bankSequence[i]) {
            allValid = 0;
          }
        }
        if(allValid) {
          printf("\n[DEBUG]: Successfully load cached bank information.\n");
          bInfo->nBanks = 1<<nFns;
          return bInfo;
        }
      }

      freeBankInformation(bInfo);
    }

    printf("\n[DEBUG]: Unable to load cached bank information, calculating them...\n");
  }

  bankInformation *bInfo = malloc(sizeof(bankInformation));
  bInfo->bankSequence = getBankSequence(&(bInfo->nBankSequence), fns, nFns);
  bInfo->nUniqueBankOffsets = 0;
  bInfo->uniqueBankOffsets = NULL;

  for(u_int64_t blockSize = 1; blockSize <= bInfo->nBankSequence; blockSize *= 2) {
    printf("[DEBUG]: Calculating parameters for block size %ld (of %ld)\r", blockSize, bInfo->nBankSequence);
    fflush(stdout);
    bInfo->nUniqueBankOffsets ++;
    bInfo->uniqueBankOffsets = realloc(bInfo->uniqueBankOffsets, bInfo->nUniqueBankOffsets * sizeof(uniqueBankOffset *));
    uniqueBankOffset *uBankOffsets = malloc(sizeof(uniqueBankOffset));
    uBankOffsets->nOffsets = 0;
    uBankOffsets->offsets = getOffsetsForUniqueDetection(bInfo->bankSequence, bInfo->nBankSequence, blockSize, 0, &(uBankOffsets->nOffsets));
    uBankOffsets->verificationOffset = getOffsetForVerification(bInfo->bankSequence, bInfo->nBankSequence, uBankOffsets->offsets, uBankOffsets->nOffsets, 1<<nFns, blockSize);
    // For small groups (e.g. size 1 and 2) there are no valid verification
    // offsets that are not already used to determine the group itself (e.g.
    // each address within the group has to be measured anyway so it is not
    // possible to measure addition addresses for verification that were not
    // measured before)
    if(uBankOffsets->verificationOffset == -1) {
      uBankOffsets->verificationOffset = 0;
    }
    uBankOffsets->sequenceLength = blockSize;

    bInfo->uniqueBankOffsets[bInfo->nUniqueBankOffsets - 1] = uBankOffsets;
  }

  printf("\n");

  bInfo->groupToBankMapping = NULL;
  bInfo->bankToGroupMapping = NULL;
  bInfo->nBanks = 1<<nFns;

  if((flags & FL_NOCACHE) == 0) {
    storeBankInformationToCache("bankInformation.txt", bInfo);
  }

  return bInfo;
}

/*******************************************************************************
 * addBankMappingInformation takes a bInfo item without mapping information and
 * measures and adds those information. That is required because the mapping
 * cannot be cached, it changes every time the initial groups are generated.
 *
 * @param bInfo: bankInformation struct without mapping information
 * @param fns: Address functions
 * @param nFns: Number of address functions
 * @param aGroups: Address groups that should be mapped (e.g. take the group
 *        index of those)
 * @param threshold: Threshold value between row hit and row conflict.
 * @param maxGroupComparisons: Maximum number of groups comparisons (e.g. how
 *        many address are compared from each group, if there are at least that
 *        many addresses in the group)
 * @param nHugePages: Number of hube page that should be allocated. Huge pages
 *        are required to get a stable mapping (e.g. 512 continuous pages)
 * @param measurementsPerAddress: Number of measurements for each address
 * @return bankInformation with group mapping information
 ******************************************************************************/
bankInformation *addBankMappingInformation(bankInformation *bInfo, u_int64_t *fns, u_int64_t nFns, addressGroups *aGroups, u_int64_t threshold, u_int64_t maxGroupComparisons, u_int64_t nHugePages, u_int64_t measurementsPerAddress) {
  while(bInfo->groupToBankMapping == NULL) {
    bInfo->groupToBankMapping = getBankMapping(aGroups, fns, nFns, threshold, maxGroupComparisons, nHugePages, measurementsPerAddress);
  }

  bInfo->bankToGroupMapping = getReverseBankMapping(bInfo->groupToBankMapping, bInfo->nBanks);

  return bInfo;
}

bankInformation *freeBankInformation(bankInformation *bInfo) {
  for(u_int64_t idx = 0; idx < bInfo->nUniqueBankOffsets; idx++) {
    free(bInfo->uniqueBankOffsets[idx]->offsets);
    free(bInfo->uniqueBankOffsets[idx]);
  }

  free(bInfo->uniqueBankOffsets);
  free(bInfo);
  return NULL;
}
