#include<linux/init.h>
#include<linux/module.h>
#include<linux/proc_fs.h>
#include<linux/slab.h>
#include<linux/kvm_host.h>

#include "allocTrace.h"

/*******************************************************************************
 * Basic information about the module
 ******************************************************************************/
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Martin Heckel");
MODULE_DESCRIPTION("Create files in procfs for tracing of the allocator mechanisms in Linux");
MODULE_VERSION("1.0");

/*******************************************************************************
 * Struct that is used to map the proc file operations to functions implemented
 * within this module.
 ******************************************************************************/
static struct proc_ops allocTraceOps = {
  .proc_read = allocTraceRead,
  .proc_write = allocTraceWrite,
  .proc_lseek = allocTraceLseek,
  .proc_open = allocTraceOpen,
  .proc_release = allocTraceRelease
};

/*******************************************************************************
 * Some global variables
 ******************************************************************************/
static allocTraceFileInfo **files = NULL;
static int nFiles = 0;
static struct proc_dir_entry *parent;

/*******************************************************************************
 * getZones iterates over the memory zones in the system and returns a list of
 * the zones that were found
 *
 * @param nZones: Pointer to store the number of zones found
 * @return List of zones
 ******************************************************************************/
struct zone **getZones(int *nZones) {
  *nZones = 0;
  struct zone **zones = NULL;

  int zoneIdx = 0;

  pg_data_t *pgdat = NODE_DATA(zoneIdx);
  while(pgdat != NULL) {
    for (int zone_type = 0; zone_type < MAX_NR_ZONES; zone_type++) {
      struct zone *zone = &pgdat->node_zones[zone_type];
      if(zone != NULL) {
        *nZones += 1;
        zones = krealloc(zones, *nZones * sizeof(struct zone *), GFP_KERNEL);
        zones[*nZones - 1] = zone;

        printk(KERN_INFO "[allocTrace]: Zone %s High: %d Batch: %d.", zone->name, zone->pageset_high, zone->pageset_batch);

        struct per_cpu_pages *pcp;
        int cpu;
        for_each_possible_cpu(cpu) {
          pcp = per_cpu_ptr(zone->per_cpu_pageset, cpu);
        }
      }
    }

    zoneIdx++;
    pgdat = NODE_DATA(zoneIdx);
  }
  return zones;
}

/*******************************************************************************
 * initProcFsDirs creates the directory structure in procfs by fetching a list
 * of zones (using getZones) and iterating over the migration type, as well as
 * orders, or logical CPU numbers (depending if the files are for the buddy
 * allocator or the per-cpu lists)
 *
 * @return On error, -1 is returned. 0 is returned on success.
 ******************************************************************************/
int initProcFsDirs(void) {
  int nZones = 0;
  struct zone **zones = getZones(&nZones);

  parent = proc_mkdir("allocTrace", NULL);
  if(parent == NULL) {
    printk(KERN_INFO, "[allocTrace]: Unable to create procfs folder /proc/allocTrace.");
    return -1;
  }

  for(int nZone = 0; nZone < nZones; nZone++) {
    struct proc_dir_entry *zoneDir = proc_mkdir(zones[nZone]->name, parent);

    if(zoneDir == NULL) {
      printk(KERN_INFO, "[allocTrace]: Unable to create procfs folder /proc/allocTrace/%s.", zones[nZone]->name);
      return -1;
    }

    for(int migrateType = 0; migrateType < MIGRATE_TYPES; migrateType++) {
      char *migDirName = NULL;
      switch(migrateType) {
        case MIGRATE_UNMOVABLE:
          migDirName = "MIGRATE_UNMOVABLE";
          break;
        case MIGRATE_MOVABLE:
          migDirName = "MIGRATE_MOVABLE";
          break;
        case MIGRATE_RECLAIMABLE:
          migDirName = "MIGRATE_RECLAIMABLE";
          break;
        case MIGRATE_PCPTYPES:
          migDirName = "MIGRATE_PCPTYPES";
          break;
        case MIGRATE_CMA:
          migDirName = "MIGRATE_CMA";
          break;
        case MIGRATE_ISOLATE:
          migDirName = "MIGRATE_ISOLATE";
          break;
        default:
          migDirName = "unknown";
      }

      struct proc_dir_entry *migrateTypeDir = proc_mkdir(migDirName, zoneDir);

      if(migrateTypeDir == NULL) {
        printk(KERN_INFO, "[allocTrace]: Unable to create procfs folder /proc/allocTrace/%s/%s.", zones[nZone]->name, migDirName);
        return -1;
      }

      char *buddyName = "buddy_pages";
      struct proc_dir_entry *buddyDir = proc_mkdir(buddyName, migrateTypeDir);

      if(buddyDir == NULL) {
        printk(KERN_INFO, "[allocTrace]: Unable to create procfs folder /proc/allocTrace/%s/%s.", zones[nZone]->name, buddyName);
        return -1;
      }

      char *pcpName = "per_cpu_pages";
      struct proc_dir_entry *pcpDir = proc_mkdir(pcpName, migrateTypeDir);

      if(pcpDir == NULL) {
        printk(KERN_INFO, "[allocTrace]: Unable to create procfs folder /proc/allocTrace/%s/%s.", zones[nZone]->name, pcpName);
        return -1;
      }



      // Little hacky to get order name (always two digit with leading 0, should
      // work up to 99 orders that are not used in any of the systems used for the
      // experiments.
      char orderName[3];
      orderName[2] = 0;
      for(int order = 0; order < MAX_ORDER; order++) {
        orderName[0] = '0' + order / 10;
        orderName[1] = '0' + order % 10;

        allocTraceFileInfo *file = kmalloc(sizeof(allocTraceFileInfo), GFP_KERNEL);

        file->offset = 0;
        file->openCount = 0;
        file->parent = buddyDir;
        file->zone = zones[nZone];
        file->order = order;
        file->migrateType = migrateType;
        file->pcpList = 0;

        struct proc_dir_entry *newFile = proc_create_data(orderName, 0644, file->parent, &allocTraceOps, file);

        file->proc = newFile;

        nFiles++;
        files = krealloc(files, nFiles * sizeof(allocTraceFileInfo*), GFP_KERNEL);
        files[nFiles-1] = file;
      }

      struct per_cpu_pages *pcp;
      int cpu;

      char cpuName[3];
      cpuName[2] = 0;
      for_each_possible_cpu(cpu) {
        cpuName[0] = '0' + cpu / 10;
        cpuName[1] = '0' + cpu % 10;

        allocTraceFileInfo *file = kmalloc(sizeof(allocTraceFileInfo), GFP_KERNEL);

        file->offset = 0;
        file->openCount = 0;
        file->parent = pcpDir;
        file->zone = zones[nZone];
        file->order = cpu;
        file->migrateType = migrateType;
        file->pcpList = 1;

        struct proc_dir_entry *newFile = proc_create_data(cpuName, 0644, file->parent, &allocTraceOps, file);

        file->proc = newFile;

        nFiles++;
        files = krealloc(files, nFiles * sizeof(allocTraceFileInfo*), GFP_KERNEL);
        files[nFiles-1] = file;
      }
    }
  }

  return 0;
}

/*******************************************************************************
 * clearProcFsDirs removes the directories that were added with initProcFsDirs()
 * so the module can be unloaded in a clean way.
 *
 * @return On error, -1 is returned. On success, 0 is returned.
 ******************************************************************************/
int clearProcFsDirs(void) {
  for(int i = 0; i < nFiles; i++) {
    if(files[i]->openCount != 0) {
      printk(KERN_INFO "[allocTrace]: Warning: File /proc/allocTrace/%s/%02d is open %d times.", files[i]->zone->name, files[i]->order, files[i]->openCount);
    }
    kfree(files[i]);
  }

  proc_remove(parent);
  kfree(files);
  nFiles = 0;
  files = NULL;

  return 0;
}

/*******************************************************************************
 * allocTraceRead implements the read operation on the files created in procfs
 * before. When the file was opened from userspace and is read, this function is
 * called.
 *
 * @param f: File that is read
 * @param buf: Buffer where the read bytes should be written to
 * @param size: Size of teh buffer, e.g. maximum number of bytes that should be
 *        read
 * @param offset: Unused in the current implementation
 * @return Number of bytes written to buf
 ******************************************************************************/
static ssize_t allocTraceRead(struct file *f, char *buf, size_t size, loff_t *offset) {
  u_int64_t pfn = 0;

  allocTraceFileInfo *file = (allocTraceFileInfo*)f->private_data;
  ssize_t read = 0;

  struct list_head *startListEntry;
  if(file->pcpList == 0) {
    startListEntry = &(file->zone->free_area[file->order].free_list[file->migrateType]);
  } else {
    startListEntry = &((per_cpu_ptr(file->zone->per_cpu_pageset, file->order))->lists[file->migrateType]);
  }

  struct list_head *currentListEntry = startListEntry;
  u_int64_t nr_free = file->zone->free_area[file->order].nr_free;
  //printk(KERN_INFO "[allocTrace]: list_head: 0x%lx, nr_free: %ld", currentListEntry, nr_free);
  off_t currentIdx = 0;

  off_t pfnIdx = file->offset / 9;
  off_t currentOffset = file->offset % 9;

  while(currentIdx < pfnIdx) {
    currentListEntry = currentListEntry->next;
    currentIdx += 1;
  }

  while(read < size && currentListEntry != NULL && (pfnIdx <= 1 || currentListEntry != startListEntry)) {
    // Each line contains one PFN in the form 0x000000\n (9 chars)

    // Dont know why but the first calculated PFN is always strange (not in the
    // range of the other PFNs) and there is always one PFN too much. For that
    // reason, the first PFN is ignored.
    pfnIdx = file->offset / 9 + 1;
    currentOffset = file->offset % 9;

    if(currentIdx != pfnIdx || (pfnIdx == 0 && pfn == 0)) {
      while(currentIdx < pfnIdx) {
        currentListEntry = currentListEntry->next;
        currentIdx += 1;
      }

      struct page *page = list_entry((void *)currentListEntry, struct page, lru);
      pfn = page_to_pfn(page);
    }

    if(pfnIdx != 0 && currentListEntry == startListEntry) {
      // Break because that was the last listEntry
      break;
    }

    if(currentOffset == 0) {
      put_user((int8_t)('0'), buf+read);
    } else if(currentOffset == 1) {
      put_user((int8_t)('x'), buf+read);
    } else if(currentOffset <= 7) {
      int8_t v = (pfn>>((5 - (currentOffset - 2)) *4)) % 16;
      if(v < 10) {
        put_user((int8_t)(v + '0'), buf+read);
      } else {
        put_user((int8_t)(v + 'a' - 10), buf+read);
      }
    } else {
      put_user((int8_t)('\n'), buf+read);
    }
    read++;
    file->offset++;
  }

  return read;
}

/*******************************************************************************
 * allocTraceWrite ignores all parameters and writes a message to dmesg that
 * writing files is not supported.
 *
 * @return -EINVAL is returned since writing to the files is not supported.
 ******************************************************************************/
static ssize_t allocTraceWrite(struct file *f, const char *buf, size_t size, loff_t *offset) {
  printk(KERN_INFO "[allocTrace]: Operation WRITE() supported.");
  return -EINVAL;
}

/*******************************************************************************
 * allocTraceLseek modifies the offset within the file according to the
 * submitted parameters.
 *
 * @param f: File that should be modified
 * @param offset: Offset that should be set
 * @param whence: Offset mode (see LSEEK(2) for more details)
 * @return New offset within the file after modification
 ******************************************************************************/
static loff_t allocTraceLseek(struct file *f, loff_t offset, int whence) {
  allocTraceFileInfo *file = (allocTraceFileInfo*)f->private_data;

  switch(whence) {
    case SEEK_SET:
      file->offset = offset;
      break;
    case SEEK_CUR:
      file->offset += offset;
      break;
    case SEEK_END:
      file->offset = file->zone->free_area[file->order].nr_free - 1;
      break;
    default:
      printk(KERN_INFO "[allocTrace]: Invalid seek whence for file /proc/allocTrace/%s/%02d.", file->zone->name, file->order);
  }
  return file->offset;
}

/*******************************************************************************
 * allocTraceOpen opens a file, sets the private_data fiels of the file to the
 * pdf_data field of the inode and increases the open counter by one. Since
 * parallel read is not supported currently, -EACCES is returned when the file
 * is already opened.
 *
 * @param ino: Inode of the file that should be opened
 * @param f: File that should be opened
 * @return On success, 0 is returned. When the file is already opened, EACCES
 *         is returned.
 ******************************************************************************/
static int allocTraceOpen(struct inode *ino, struct file *f) {
  void *ptr = pde_data(ino);
  f->private_data = ptr;

  allocTraceFileInfo *file = (allocTraceFileInfo*)f->private_data;

  if(file->openCount > 0) {
    printk(KERN_INFO "[allocTrace]: Unable to open file /proc/allocTrace/%s/%02d. File is already opened.", file->zone->name, file->order);
    return -EACCES;
  }

  file->openCount += 1;
  file->offset = 0;

  try_module_get(THIS_MODULE);
  return 0;
}

/*******************************************************************************
 * allocTraceRelease closes a file by decreasing the open counter by one. 
 *
 * @param ino: Inode of the file that should be opened
 * @param f: File that should be opened
 * @return: 0 is returned on success, -EACCES on error.
 ******************************************************************************/
static int allocTraceRelease(struct inode *ino, struct file *f) {
  allocTraceFileInfo *file = (allocTraceFileInfo*)f->private_data;
  if(file->openCount <= 0) {
    return -EACCES;
  }
  file->openCount -= 1;
  module_put(THIS_MODULE);
  return 0;
}

/*******************************************************************************
 * allocTraceInit is called when the module is loaded an triggers the
 * initialization of the procfs file structure created by the module.
 *
 * @return On success, 0 is returned.
 ******************************************************************************/
static int __init allocTraceInit(void) {
  initProcFsDirs();
  /*for(int i = 0; i < nFiles; i++) {
    printk(KERN_INFO "Loaded File 0x%lx for /proc/allocTrace/%s/%02d.", files[i], files[i]->zone->name, files[i]->order);
  }*/
  printk(KERN_INFO "[allocTrace]: Loaded");
  return 0;
}

/*******************************************************************************
 * allocTraceExit is called when the module is unloaded and triggers the
 * cleanup of the procfs file structure created by the module.
 ******************************************************************************/
static void __exit allocTraceExit(void) {
  clearProcFsDirs();
  printk(KERN_INFO "[allocTrace]: Unloaded");
}

/*******************************************************************************
 * Register init and exit functions
 ******************************************************************************/
module_init(allocTraceInit);
module_exit(allocTraceExit);
